

# https://blenderartists.org/t/scripts-create-camera-image-plane/580839


from asyncio import create_subprocess_exec
import bpy
import os
from bpy_extras.image_utils import load_image
import math

from . import imageloading
from .argutil import ImportArguments
from .Composite import CompositeKey
from . import version

MAXLIDARDEPTH=5.0
PLANENAMEPREFIX="zl_ip_"
MATNAMEPREFIX="zl_ip_mat_"
YASPECTRATIO=9.0/16.0

def srgb_to_linear_rgb(c):
    """
    Converts a color component from sRGB to linear space.
    This is the standard formula.
    """
    if c <= 0.04045:
        return c / 12.92
    else:
        return ((c + 0.055) / 1.055) ** 2.4

def ourplanename(name):
    return name.startswith(PLANENAMEPREFIX)


def setplanescale(imageplane,angle,depth):
    setobjscale(imageplane,depth,angle)

def setobjscale(obj,depth,angle):
    cA = angle
    s = depth*math.tan(cA/2)

    obj.scale.x = depth
    obj.scale.y = depth
    obj.scale.z = depth


class CreateCameraImagePlane:
    """Create image plane for camera"""
    bl_idname= "object.createcameraimageplane"
    bl_label="Camera Image Plane"
    bl_options={'REGISTER', 'UNDO'}

    ALPHARAMPWHITE = 0.202
    ALPHARAMPBLACK = 0.436

    
    def apply_texture_options(self, texture, ia, img_spec):
        CompositeKey.setResolutionFromImage(img_spec.image)
        image_user = texture.image_user
        img_spec.image.alpha_mode = 'NONE'
        image_user.use_auto_refresh = True
        image_user.frame_start = ia.exportbase.timelinestartframe
        image_user.frame_offset = img_spec.frame_offset + (ia.exportbase.startframe - 1)
        image_user.frame_duration = img_spec.frame_duration


        texture.extension = 'CLIP'  # Default of "Repeat" can cause artifacts


#  create a 3D plane, parent it to the camera, and apply the camera's video footage as a texture
    def createImagePlaneForCamera(self, context, cameraobj, ia:ImportArguments):
        #start_time = datetime.datetime.now()
        #cont = context.area.type
        #print("starting context ",str(cont))

        imageplane = None

        try:
            #create imageplane
            bpy.ops.mesh.primitive_plane_add()#radius = 0.5)
            imageplane = bpy.context.active_object
            imageplane.display.show_shadows = False
            imageplane.hide_render = False
            #print(f"imageplane {imageplane}")
            mesh = imageplane.data
            #print(f"mesh {mesh}")
            mesh.use_auto_texspace = True
            vertices = mesh.vertices

            # Translate the vertices
            xfov = ia.exportbase.camera.camera_frames[0].xFovDegrees

            Z = -1  # 1 meter

            X =  math.tan(math.radians(xfov/2))
            cam = ia.exportbase.camera
            Y = (X * cam.height) / cam.width
            #print(f"createimagelance xfov {xfov} X {X} Y {X} Z {Z}")

            vertices[0].co = (-X,-Y,Z)
            vertices[1].co = (X,-Y,Z)
            vertices[2].co = (-X,Y,Z)
            vertices[3].co = (X,Y,Z)

            for vertex in vertices:
                print(f"{vertex} {vertex.co}")

            depth = ia.exportbase.depth_amount            
            imageplane.location = (0,0,0)
            imageplane.parent = cameraobj
            setplanescale(imageplane, cameraobj.data.angle, 1.0)

            #print("imageplane.name before rename ", imageplane.name)
            imageplane.name = PLANENAMEPREFIX+ia.cameraname

            #setup material
            assert(len( imageplane.material_slots) == 0)
            bpy.ops.object.material_slot_add()
            material = bpy.data.materials.new(MATNAMEPREFIX+ia.cameraname)
            print("material name] is ",material.name)

            imageplane.material_slots[0].material = material

            material.use_nodes = True
            material.blend_method = 'CLIP'

            node_tree = material.node_tree
            nodes = material.node_tree.nodes
            links = material.node_tree.links

            nodes.clear()

            # --- Start of Node Graph Creation ---

            # Always create the output and camera texture nodes
            outnode = nodes.new('ShaderNodeOutputMaterial')
            outnode.location = (600, 0)

            camtexture = nodes.new('ShaderNodeTexImage')
            camtexture.location = (-600, 0)
            
            spec = CompositeKey.loadcameraimage(ia, ia.exportbase.media_camera.path, True, colorspace=ia.exportbase.media_camera.colorspace)
            print(f"camtexture Using spec {spec}")
            if spec:
                camtexture.image = spec.image
                camtexture.show_texture = True
                self.apply_texture_options(camtexture, ia, spec)

            if ia.exportbase.media_aimatte.path != "":
                print("Using dedicated AI matte file with a Mix Shader for material.")

                # The Principled BSDF is not ideal for this. A simple mix of Emission and
                # Transparent shaders is the correct and most efficient way to create an
                # image plane that emits light (the video) and has clean transparency.

                # 1. Create the necessary shaders
                mix_shader = nodes.new('ShaderNodeMixShader')
                mix_shader.location = (300, 0)

                transparent_bsdf = nodes.new('ShaderNodeBsdfTransparent')
                transparent_bsdf.location = (100, -150)

                emission_shader = nodes.new('ShaderNodeEmission')
                emission_shader.location = (100, 150)

                # 2. Create and configure the AI Matte texture node
                mattetexture = nodes.new('ShaderNodeTexImage')
                mattetexture.label = "AIMatte"
                mattetexture.location = (-400, 400)
                spec = CompositeKey.loadcameraimage(ia, ia.exportbase.media_aimatte.path, True)

                if spec:
                    mattetexture.image = spec.image
                    # CRITICAL: Matte data is linear data, not color. Set to 'Non-Color'.
                    mattetexture.image.colorspace_settings.name = 'Non-Color'
                    mattetexture.image.alpha_mode = 'NONE'
                    mattetexture.show_texture = True
                    self.apply_texture_options(mattetexture, ia, spec)

                    # 3. Link the nodes together
                    # Link the camera video to the Emission shader's color
                    links.new(camtexture.outputs['Color'], emission_shader.inputs['Color'])

                    # Link the AI matte's color output to the Mix Shader's factor.
                    # A B&W image's color output works perfectly as a factor.
                    links.new(mattetexture.outputs['Color'], mix_shader.inputs['Fac'])

                    # A standard matte is black (0) for transparent and white (1) for opaque.
                    # Mix Shader Top Input (socket 0) is used when Fac=0.
                    # Mix Shader Bottom Input (socket 1) is used when Fac=1.
                    # Therefore:
                    # - Top (0) should be Transparent BSDF (for black parts of matte).
                    # - Bottom (1) should be Emission Shader (for white parts of matte).

                    # The API uses indexed inputs for shaders.
                    shader_inputs = [s for s in mix_shader.inputs if s.type == 'SHADER']
                    if len(shader_inputs) == 2:
                        links.new(transparent_bsdf.outputs['BSDF'], shader_inputs[0])
                        # CORRECTED LINE: The Emission node's output is named 'Emission'.
                        links.new(emission_shader.outputs['Emission'], shader_inputs[1])
                    else: # Fallback for older Blender versions if socket names differ
                         links.new(transparent_bsdf.outputs['BSDF'], mix_shader.inputs[1])
                         # CORRECTED LINE: The Emission node's output is named 'Emission'.
                         links.new(emission_shader.outputs['Emission'], mix_shader.inputs[2])

                    # Link the final result to the material output
                    links.new(mix_shader.outputs['Shader'], outnode.inputs['Surface'])
                else:
                    # Fallback if matte fails to load: just show the video unmasked.
                    emission_shader = nodes.new('ShaderNodeEmission')
                    emission_shader.location = (300, 0)
                    links.new(camtexture.outputs['Color'], emission_shader.inputs['Color'])
                    links.new(emission_shader.outputs['Emission'], outnode.inputs['Surface'])
            else:
                # No dedicated matte, use the Green Screen keyer graph from the image
                print("Using ColorKeyer graph for material.")
                groupname = CompositeKey.kColorKeyerName
                if groupname not in bpy.data.node_groups:
                    print(f"Error: Node group '{groupname}' not found. Cannot create material.")
                    return {'CANCELLED'}

                # 1. Create nodes from the image graph
                colorkeyer = nodes.new(type='ShaderNodeGroup')
                colorkeyer.node_tree = bpy.data.node_groups[groupname]
                colorkeyer.location = (-350, 0)

                keyer_settings = ia.exportbase.keyer if hasattr(ia.exportbase, 'keyer') and ia.exportbase.keyer else None
                if keyer_settings:
                    print("Applying keyer settings from export base.")
                    # Key Color
                    if 'KeyColor' in colorkeyer.inputs:
                        # The keyer node operates in linear color space. User-provided keyer_settings
                        # are in sRGB, so we must convert them to linear for correct results.
                        
                        # Convert each color component from sRGB to linear
                        #linear_r = srgb_to_linear_rgb(keyer_settings.KeyColorRed)
                        #linear_g = srgb_to_linear_rgb(keyer_settings.KeyColorGreen)
                        #linear_b = srgb_to_linear_rgb(keyer_settings.KeyColorBlue)
                        
                        colorkeyer.inputs['KeyColor'].default_value = (
                            keyer_settings.KeyColorRed,
                            keyer_settings.KeyColorGreen,
                            keyer_settings.KeyColorBlue,
                            1.0
                        )
                    # Map other settings, handling name differences.
                    if True:
                        if 'AlphaThreshold' in colorkeyer.inputs and hasattr(keyer_settings, 'AlphaThreshold'):
                            colorkeyer.inputs['AlphaThreshold'].default_value = keyer_settings.AlphaThreshold
                        if 'AlphaOffset' in colorkeyer.inputs and hasattr(keyer_settings, 'AlphaOffset'):
                            colorkeyer.inputs['AlphaOffset'].default_value = keyer_settings.AlphaOffset
                        if 'BlackClip' in colorkeyer.inputs and hasattr(keyer_settings, 'ClipBlack'):
                            colorkeyer.inputs['BlackClip'].default_value = keyer_settings.ClipBlack
                        if 'WhiteClip' in colorkeyer.inputs and hasattr(keyer_settings, 'ClipWhite'):
                            colorkeyer.inputs['WhiteClip'].default_value = keyer_settings.ClipWhite * 100
                            
                        # NOTE: This is an assumption based on the Keyer class structure.
                        # 'WeightRed' from the Keyer class is mapped to the 'RedWeight' input.
                        if 'RedWeight' in colorkeyer.inputs and hasattr(keyer_settings, 'WeightRed'):
                            colorkeyer.inputs['RedWeight'].default_value = keyer_settings.WeightRed
                        # 'WeightOther' is mapped to both 'GreenWeight' and 'BlueWeight'.
                        isgreen = True
                        if hasattr(keyer_settings, 'GreenBlue'):
                            isgreen = keyer_settings.GreenBlue
                        if isgreen and 'GreenWeight' in colorkeyer.inputs and hasattr(keyer_settings, 'WeightOther'):
                            colorkeyer.inputs['GreenWeight'].default_value = keyer_settings.WeightOther
                        if not isgreen and 'BlueWeight' in colorkeyer.inputs and hasattr(keyer_settings, 'WeightOther'):
                            colorkeyer.inputs['BlueWeight'].default_value = keyer_settings.WeightOther
                else:
                    # Set the default keying color to a natural green (hex #3B6D48FF).
                    print("No keyer settings found, using default green.")
                    colorkeyer.inputs['KeyColor'].default_value = (0.088, 0.286, 0.080, 1.0)

                alphalevels = nodes.new('ShaderNodeValToRGB') # This is the ColorRamp node
                alphalevels.label = "AlphaLevels"
                alphalevels.location = (-100, 200)

                color_ramp = alphalevels.color_ramp
                if len(color_ramp.elements) >= 2:
                        # Configure the first stop (left side)
                    color_ramp.elements[0].position = CreateCameraImagePlane.ALPHARAMPWHITE
                    color_ramp.elements[0].color = (1.0, 1.0, 1.0, 1.0)  # White

                    # Configure the second stop (right side)
                    color_ramp.elements[1].position = CreateCameraImagePlane.ALPHARAMPBLACK
                    color_ramp.elements[1].color = (0.0, 0.0, 0.0, 1.0)  # Black

                transparent_bsdf = nodes.new('ShaderNodeBsdfTransparent')
                transparent_bsdf.location = (150, 200)

                emission = nodes.new('ShaderNodeEmission')
                emission.location = (150, -100)
                emission.inputs['Strength'].default_value = 1.0

                add_shader = nodes.new('ShaderNodeAddShader')
                add_shader.location = (400, 50)

                # 2. Link the nodes together
                # Link camera texture to the keyer
                links.new(camtexture.outputs['Color'], colorkeyer.inputs['GreenScreen'])

                # Link keyer outputs
                links.new(colorkeyer.outputs['RGB'], emission.inputs['Color'])
                links.new(colorkeyer.outputs['Alpha'], alphalevels.inputs['Fac'])

                # Link alpha processing to transparent shader's color
                links.new(alphalevels.outputs['Color'], transparent_bsdf.inputs['Color'])

                # Link shaders into the Add Shader
                links.new(transparent_bsdf.outputs['BSDF'], add_shader.inputs[0]) # Top socket
                links.new(emission.outputs['Emission'], add_shader.inputs[1])  # Bottom socket

                # Link final shader to the output
                links.new(add_shader.outputs['Shader'], outnode.inputs['Surface'])


            # --- End of Node Graph Creation ---

            # Do this last so we can edit the object
            # Causes problems. Can't show later in Blender
            #imageplane.hide_viewport = True

        except Exception as e: 
            imageplane.select_set(False)
            cameraobj.select_set(True)
            raise e    
        return {'FINISHED'}

    #def execute(self, context):
    #    camera = bpy.context.active_object #bpy.data.objects['Camera']
    #    if camera.type == None or camera.type != 'CAMERA':
    #        self.report({'INFO'}, "Selected object not CAMERA!")
    #        return {'FINISHED'}
    #    else:
    #        return self.createImagePlaneForCamera(context,camera)