#import numpy as np

import time
from xmlrpc.client import Boolean
import bpy
import os
from .argutil import ImportArguments,SceneType


RENDERSCENE="Render"

def setup_render_path():
    """
    Reads the renderdir property, creates the directory if needed, and sets
    the scene's final render filepath.
    This safely handles Blender's relative '//' paths.
    """
    # 1. Get the path from the property. It might be relative (e.g., '//../Render')
    render_directory_prop = bpy.context.scene.AFTERBURNER.renderdir

    if not render_directory_prop:
        print("Error: Render directory property is not set.")
        return False

    # 2. Use Blender's path utility to convert it to an absolute path for the OS
    abs_render_directory = bpy.path.abspath(render_directory_prop)

    try:
        # 3. Create the directory using the resolved absolute path.
        os.makedirs(abs_render_directory, exist_ok=True)
    except OSError as e:
        print(f"Error: Could not create render directory '{abs_render_directory}'. Reason: {e}")
        return False

     # Ensure the path for Blender is relative if the original was.
    # bpy.path.relpath will return an absolute path if it cannot be made relative.
    blender_render_path = bpy.path.relpath(render_directory_prop)
    if not os.path.isabs(blender_render_path):
        blender_render_path = f"//{blender_render_path}"

    # 4. Construct the filename pattern using the original property for the base name.
    prefix_base = os.path.basename(blender_render_path.replace('//',''))
    filename_pattern = f"{prefix_base}_#####.exr"

    # 5. For Blender's render engine, use the original (potentially relative) path.
    full_render_filepath = os.path.join(blender_render_path, filename_pattern)
    
    bpy.context.scene.render.filepath = full_render_filepath
    print(f"Set render output path to: {bpy.context.scene.render.filepath}")
    return True


class Daily:
        def __init__(self,ia:ImportArguments):
            self.uuid = ""
            self.ia = ia

        def setup(self,deletecamera:bool):
            """
            
            """

            if deletecamera:
                camera = bpy.data.objects.get(self.ia.cameraname)
                if camera != None:
                    print(f"Found {self.ia.cameraname} to delete")
                    for obj in camera.children:
                        print(f"Remove child {obj.name}")
                        bpy.data.objects.remove(obj, do_unlink=True)
                    bpy.data.objects.remove(camera, do_unlink=True)

        def setourcamera(self):
            camera = bpy.data.objects.get(self.ia.cameraname)
            print(f"Camera is {camera}")
            scene = bpy.context.scene
            if camera:
                scene.camera = camera
            region = next(iter([area.spaces[0].region_3d for area in bpy.context.screen.areas if area.type == 'VIEW_3D']), None)
            if region:
                region.view_perspective = 'CAMERA'

        def finish(self):
            """
            After import leave us looking through new camera
            """
            self.setourcamera()
            # Need to see camera texture
            # For workbench
            bpy.context.scene.display.shading.color_type = 'TEXTURE'
            #if bpy.context.scene.AFTERBURNER.saveblend_boolean:

            filepath = bpy.context.scene.AFTERBURNER.blendfilepath

            print('Before save blender file')
            if os.path.exists(filepath):
                os.remove(filepath)

            self.normalrendersetup()
            
            bpy.ops.wm.save_as_mainfile(filepath=filepath, copy= True, check_existing=False)
            print("Saved blender file:",filepath)
            
            #self.fastrender()
            #bpy.ops.wm.save_as_mainfile(filepath=filepath, copy= True, check_existing=False)

        def setshading(self):
                # XXX Look at this. Why is it here. Why is it after save. We set material preview in another location
            # for eevee  
            for area in bpy.context.screen.areas: 
                if area.type == 'VIEW_3D':
                    for space in area.spaces: 
                        if space.type == 'VIEW_3D':
                            print(f"Setting 'RENDERED' instead of {space.shading.type} ")
                            space.shading.type = 'RENDERED'

        def normalrendersetup(self):
            self.setrenderingdefaults(False)

            setup_render_path()
    

        def fastrender(self):
            self.setourcamera()
            self.setrenderingdefaults(True)

            if bpy.context.scene.AFTERBURNER.superpath != "":
                render_path = bpy.context.scene.AFTERBURNER.superpath
                # Ensure the path for Blender is relative if the original was.
                # This check ensures we don't add "//" to an already absolute path.
                if not os.path.isabs(render_path):
                     # Check if it already starts with "//"
                    if not render_path.startswith('//'):
                        render_path = f"//{render_path}"
                bpy.context.scene.render.filepath = render_path
                
                # For the fastest possible preview, prioritize Legacy EEVEE first as it has
                # less overhead for simple scenes. Fall back to EEVEE_NEXT for future compatibility.
                try:
                    # Try setting the render engine to the faster Legacy EEVEE
                    bpy.context.scene.render.engine = 'BLENDER_EEVEE'
                    print("Render engine set to 'BLENDER_EEVEE' for maximum speed.")
                except TypeError:
                    # If Legacy EEVEE is ever removed, fall back to EEVEE_NEXT
                    print("Legacy EEVEE not found. Falling back to EEVEE_NEXT.")
                    try:
                        bpy.context.scene.render.engine = 'BLENDER_EEVEE_NEXT'
                        print("Render engine set to 'BLENDER_EEVEE_NEXT'.")
                    except TypeError:
                        print("Could not set render engine to any EEVEE version.")

                # Use opengl for interactive mode, and standard render for background mode
                if bpy.app.background:
                    print("Executing Eevee render for animation (background mode)...")
                    bpy.ops.render.render(animation=True)
                else:
                    print("Executing OpenGL viewport render for animation (interactive mode)...")
                    bpy.ops.render.opengl('INVOKE_DEFAULT', animation=True)



# In daily.py

        def setrenderingdefaults(self,fast:Boolean):
            if RENDERSCENE in bpy.data.scenes:
                scene = bpy.data.scenes[RENDERSCENE]
            else:
                scene = bpy.context.scene

            if not scene:
                return

            if fast:
                # --- FAST PREVIEW SETTINGS ---
                scene.render.image_settings.file_format = 'FFMPEG'
                scene.render.ffmpeg.format = 'MPEG4'
                scene.render.ffmpeg.audio_codec = 'AAC'
                scene.render.ffmpeg.audio_channels = 'MONO'
                scene.render.use_overwrite = True
                scene.render.use_placeholder = False
                scene.render.use_file_extension = False
                scene.render.image_settings.color_depth = '8'

                # FIX #1: BYPASS THE COMPOSITOR
                print("set render.use_compositing to false")
                scene.render.use_compositing = False

                # FIX #2: ENSURE VIEW TRANSFORM IS 'Standard'
                scene.view_settings.view_transform = 'Standard'

                # FIX #3: DISABLE EEVEE EFFECTS & SHADOWS, PRESERVING LEGACY CHECKS
                if hasattr(scene, "eevee"):
                    print("Optimizing EEVEE for speed, preserving legacy compatibility.")

                    # --- THIS IS THE CORRECT FIX ---
                    # FIX #4: DISABLE MASTER SHADOWS USING THE CORRECT PROPERTY
                    # This directly corresponds to the 'Shadows' checkbox in the UI.
                    if hasattr(scene.eevee, 'use_shadows'):
                        print("Disabling Eevee master shadows via use_shadows.")
                        scene.eevee.use_shadows = False

                    # Ambient Occlusion (Primary cause of darkness)
                    if hasattr(scene.eevee, 'gtao_enabled'):
                        print("set gtao_enabled to false")
                        scene.eevee.gtao_enabled = False
                    elif hasattr(scene.eevee, 'use_gtao'): # Legacy
                        print("legacy set use_gtao to false")
                        scene.eevee.use_gtao = False

                    # Bloom
                    if hasattr(scene.eevee, 'use_bloom'): # Legacy and Modern
                        scene.eevee.use_bloom = False

                    # Screen Space Reflections
                    if hasattr(scene.eevee, 'ssr_enabled'):
                        scene.eevee.ssr_enabled = False
                    elif hasattr(scene.eevee, 'use_ssr'): # Legacy
                        scene.eevee.use_ssr = False

                    # Samples (2 is fine, does not affect brightness)
                    if hasattr(scene.eevee, 'taa_render_samples'):
                        scene.eevee.taa_render_samples = 16
                    elif hasattr(scene.eevee, 'samples'): # Legacy
                        scene.eevee.samples = 16

                    # --- Your original shadow quality settings are still relevant for the 'else' block ---
                    if hasattr(scene.eevee, 'use_soft_shadows'):
                        scene.eevee.use_soft_shadows = False
                    if hasattr(scene.eevee, 'shadow_cube_size'):
                        scene.eevee.shadow_cube_size = '512'
                    if hasattr(scene.eevee, 'shadow_cascade_size'):
                        scene.eevee.shadow_cascade_size = '1024'
                    print(f"scene.eevee {scene.eevee}")
            else:
                # --- NORMAL (QUALITY) RENDER SETTINGS ---

                # Re-enable everything for quality renders
                scene.render.use_compositing = True
                scene.view_settings.view_transform = 'Filmic' # Use Filmic for quality renders

                if hasattr(scene,"eevee"):
                    # --- RE-ENABLE MASTER SHADOWS ---
                    if hasattr(scene.eevee, 'use_shadows'):
                        scene.eevee.use_shadows = True

                    # Restore other quality settings, respecting legacy/modern properties
                    if hasattr(scene.eevee, 'gtao_enabled'):
                        scene.eevee.gtao_enabled = True
                    elif hasattr(scene.eevee, 'use_gtao'): # Legacy
                        scene.eevee.use_gtao = True

                    if hasattr(scene.eevee, 'ssr_enabled'):
                        scene.eevee.ssr_enabled = True
                    elif hasattr(scene.eevee, 'use_ssr'): # Legacy
                        scene.eevee.use_ssr = True

                    if hasattr(scene.eevee, 'taa_render_samples'):
                        scene.eevee.taa_render_samples = 64
                    elif hasattr(scene.eevee, 'samples'): # Legacy
                        scene.eevee.samples = 64

                # Set quality file format
                # example https://www.youtube.com/watch?v=yO5xCOoAGqg
                # Blender 5.0+ removed OPEN_EXR_MULTILAYER enum, use OPEN_EXR with use_multilayer property
                scene.render.image_settings.file_format = 'OPEN_EXR'
                scene.render.image_settings.color_mode = 'RGBA'
                scene.render.image_settings.exr_codec = 'DWAA'
                scene.render.image_settings.color_depth = '16'
                # Enable multilayer if the property exists (Blender 5.0+)
                if hasattr(scene.render.image_settings, 'use_multilayer'):
                    scene.render.image_settings.use_multilayer = True
                scene.render.use_overwrite = True
                scene.render.use_placeholder = False # For now during testing

            # These settings apply to both modes
            scene.render.film_transparent = True
            scene.view_settings.look = 'None'
