# Make Assets
# Copyright Lightcraft Technology 2022

import bpy
import os
import re
import time
from typing import Dict, Tuple

from bpy.types import Material, Object

from .utilfunctions import (
    convert_name,
    get_props,
    get_prop_types,
    get_relevant_child_obj,
    get_parent_from_selected,
    get_mesh_links,   # <- we now rely on the USD references mapping
)


# Helpers
def _canon(s: str) -> str:
    """Canonicalize names for matching: trim, collapse whitespace to underscores."""
    s = s.strip()
    s = re.sub(r"\s+", "_", s)
    s = re.sub(r"_+", "_", s)
    return s


def _filename_base(path: str) -> str:
    return os.path.splitext(os.path.basename(path))[0]


# Material dedupe (unchanged)
def material_signature(mat: bpy.types.Material | None) -> Tuple:
    if mat is None:
        return ("NONE",)

    if not mat.use_nodes or not mat.node_tree:
        return ("NO_NODES", tuple(round(c, 5) for c in mat.diffuse_color))

    nt = mat.node_tree
    node_counts: Dict[str, int] = {}
    images, groups = set(), set()

    for n in nt.nodes:
        node_counts[n.bl_idname] = node_counts.get(n.bl_idname, 0) + 1
        if n.type == 'TEX_IMAGE' and n.image:
            images.add((n.image.name_full, n.image.colorspace_settings.name))
        if n.bl_idname == 'ShaderNodeGroup' and n.node_tree:
            groups.add(n.node_tree.name_full)

    links = sorted(
        (l.from_node.bl_idname, l.from_socket.identifier,
         l.to_node.bl_idname,   l.to_socket.identifier)
        for l in nt.links
    )

    return (
        tuple(sorted(node_counts.items())),
        tuple(links),
        tuple(sorted(images)),
        tuple(sorted(groups)),
    )

def dedupe_object_materials(obj: Object, mat_dict: Dict[Tuple, Material]):
    """Replace duplicate materials on object's slots using mat_dict."""
    for slot in obj.material_slots:
        mat = slot.material
        if mat is None:
            continue
        key = material_signature(mat)
        existing = mat_dict.get(key)
        if existing is None:
            mat_dict[key] = mat
        else:
            if existing is not mat:
                slot.material = existing
                if mat.users == 0:
                    bpy.data.materials.remove(mat)


# TopLevel import (kept, but only for optional inspection; not used for matching)
def import_top_level(cur_dir: str):
    """Import a top-level .usd into a 'TopLevel' scene; names here are NOT used for matching."""
    prev_scene = bpy.context.window.scene
    scene = bpy.data.scenes.new("TopLevel")
    bpy.context.window.scene = scene

    for filename in os.listdir(cur_dir):
        if not filename.endswith(".usd"):
            continue

        bpy.ops.wm.usd_import(filepath=os.path.join(cur_dir, filename))
        # Make names unique; TopLevel names are irrelevant to asset selection
        for obj in scene.collection.all_objects:
            if isinstance(obj, bpy.types.Object):
                try:
                    obj.name = obj.name + "_|_"
                except Exception:
                    pass
        break

    bpy.context.window.scene = prev_scene


# Core: build assets using USD reference mapping
def import_props(prop_types):
    """
    Import USDs as assets. Selection is driven by USD references found in linked_meshes.json:
    - We build a set of referenced filename-bases (underscored).
    - Only USDs whose filename-bases (underscored) are referenced are turned into assets.
    - TopLevel scene object NAMES are ignored for matching.
    """
    layer_collection = bpy.context.view_layer.layer_collection
    material_dict: Dict[Tuple, Material] = {}

    # Build the referenced set from USD links (values are filename bases).
    cur_dir = os.path.dirname(bpy.data.filepath)
    mesh_links = get_mesh_links(cur_dir)  # { "<parent/child or child>": "<filename_base>" }
    referenced_keys = {_canon(v) for v in mesh_links.values()}

    for prop_path, prop_type, _legacy_name in prop_types:
        prop_key = _filename_base(prop_path)          # e.g., 'SM_Base_Engle_Modul_2_broken'
        prop_key_canon = _canon(prop_key)

        # Only create assets that are actually referenced by the level USD(s)
        if prop_key_canon not in referenced_keys:
            continue

        # Import the USD
        bpy.ops.object.select_all(action='DESELECT')
        bpy.ops.wm.usd_import(
            filepath=prop_path,
            create_collection=True if prop_type == 1 else False,
        )

        imported_root_obj = get_parent_from_selected()
        obj = get_relevant_child_obj(imported_root_obj)
        if obj is None:
            # print(f"[WARN] Could not find relevant object for {prop_path}")
            continue

        # Normalize scale: USD cm -> Blender m (if needed)
        if obj.scale.x == 1 or obj.scale.y == 1 or obj.scale.z == 1:
            obj.scale.x /= 100
            obj.scale.y /= 100
            obj.scale.z /= 100

        if prop_type == 1:
            # Collection asset
            col = None
            for child_col in bpy.context.scene.collection.children:
                if isinstance(child_col, bpy.types.Collection):
                    if child_col.name == bpy.context.view_layer.active_layer_collection.name:
                        col = child_col
                        break
            if col is None:
                # print(f"[WARN] Imported collection not found for {prop_path}")
                continue

            # Canonical, USD-safe name for the asset
            col.name = prop_key

            # Dedupe materials within the imported collection
            for obj_c in list(col.all_objects):
                if hasattr(obj_c, "material_slots"):
                    dedupe_object_materials(obj_c, material_dict)

            # Hide import helper empty and mark asset
            try:
                bpy.context.object.empty_display_size = 0
            except Exception:
                pass

            col.asset_mark()
            col.asset_generate_preview()

            # Return active collection to Scene
            bpy.context.view_layer.active_layer_collection = layer_collection

        else:
            # Single-object asset
            obj.name = prop_key

            # Dedupe materials on this object
            dedupe_object_materials(obj, material_dict)

            try:
                bpy.context.object.empty_display_size = 0
            except Exception:
                pass

            obj.asset_mark()
            obj.asset_generate_preview()

    # Finalize
    bpy.context.scene.name = "Assets"

    # Optional cleanup: remove TopLevel scene if it exists (not needed anymore)
    if "TopLevel" in bpy.data.scenes:
        bpy.data.scenes.remove(bpy.data.scenes["TopLevel"], do_unlink=True)

    # Remove unused datablocks
    bpy.ops.outliner.orphans_purge(do_local_ids=True, do_linked_ids=True, do_recursive=True)


# Entry point
def make_assets(cur_dir: str):
    t0 = time.perf_counter()
    print(f"\nMaking assets in {cur_dir}...")

    # Gather all candidate USDs from Assets/ (we'll filter by references next)
    props = get_prop_types(get_props(cur_dir))

    # Import top-level USD for optional inspection (names not used for matching)
    import_top_level(cur_dir)

    # Build assets strictly from USD references (linked_meshes.json)
    import_props(props)

    t1 = time.perf_counter()
    print(f"Total time: {round(t1 - t0, 3)} seconds")
