# Make Scene
# Copyright Lightcraft Technology 2022
import bpy
import os
import json
import time

from mathutils import Matrix
from .utilfunctions import convert_name, get_props, get_prop_types, clear_parent_and_keep_transform, is_mesh_layer, get_mesh_links

from pxr import Usd, UsdGeom


def rescale_point_clouds(divide_by):
    """Rescale all point clouds in the scene by dividing their coordinates by the given value"""
    for obj in bpy.context.view_layer.objects:
        if obj.type != 'POINTCLOUD':
            continue
        if not obj.data or not obj.data.points:
            continue
        for point in obj.data.points:
            point.co /= divide_by


def fix_missing_scope_types(usd_path, save_path):
    stage = Usd.Stage.Open(usd_path)
    root_prim = stage.GetPrimAtPath("/Root")

    if not root_prim:
        raise RuntimeError("No /Root prim found in USD file.")

    modified = False

    for child in root_prim.GetChildren():
        path = child.GetPath()
        type_name = child.GetTypeName()
        
        if not type_name:
            UsdGeom.Scope.Define(stage, path)
            print(f"Set type Scope for: {path}")
            modified = True
        else:
            print(f"Skipped {path} (already has type '{type_name}')")

    if modified:
        stage.GetRootLayer().Export(save_path)
        print(f"Saved modified USD to: {save_path}")
    else:
        print("No changes made. All children already have types.")


def divide_light_powers(divide_by):
    """Divide powers of each light object in the scene by the given value"""
    for obj in bpy.context.view_layer.objects:
        if obj.type != 'LIGHT':
            continue
        obj.data.energy /= divide_by


def import_empties(props, cur_dir, mesh_links):
    """Import empties from USD files and return a list of empties with their asset names and types"""
    # Create a new scene to place empties (and other objects like lights) into
    DEBUG = False

    empty_scene = bpy.data.scenes.new("Empties")
    
    # Set that scene to active
    bpy.context.window.scene = empty_scene
    
    # Find top level .usd file with empties
    empty_file_path = None
    for filename in os.listdir(os.path.join(cur_dir)):
        if filename.endswith(".usd"):
            empty_file_path = os.path.join(cur_dir, filename)

            existing_objects = set(bpy.context.scene.objects)

            # Import empties from file
            bpy.ops.wm.usd_import(
                filepath=empty_file_path,
                create_collection=False,
                import_materials=False,
                import_meshes=False,
                import_subdiv=False,
                import_volumes=False,
            )

            imported_objects = set(bpy.context.scene.objects) - existing_objects
            if not imported_objects:
                return []

            empties = bpy.context.scene.objects

            empties_json_path = os.path.join(cur_dir, "empties.json")
            props_json_path   = os.path.join(cur_dir, "props.json")

            empties_names = [convert_name(empty_obj.name + "_|_") for empty_obj in empties]
            props_names = props

            if DEBUG:
                with open(empties_json_path, "w", encoding = "utf-8") as f:
                    json.dump(empties_names, f, indent = 4, ensure_ascii = False)

                with open(props_json_path, "w", encoding = "utf-8") as f:
                    json.dump(props_names, f, indent = 4, ensure_ascii = False)

            # Filter out empties that are not props
            empties_filtered = []
            empties_matched = []

            for empty_obj in empties:
                # Divide the location values by 100
                empty_obj.location.x /= 100
                empty_obj.location.y /= 100
                empty_obj.location.z /= 100

                empty_obj.name = empty_obj.name + "_|_"
                name_ = convert_name(empty_obj.name)
                if empty_obj.parent:
                    parent_name_ = convert_name(empty_obj.parent.name)
                    reference_name = mesh_links.get(f"{parent_name_}/{name_}", "")
                else:
                    reference_name = mesh_links.get(name_, "")

                for prop in props:
                    if reference_name:
                        if reference_name != prop[2]:
                            continue
                    else:
                        if name_ != prop[2]:
                            continue
                    empties_matched.append([prop[2], prop[1], empty_obj.name])  # asset name, asset type, empty name
                    empties_filtered.append([prop[2], prop[1], empty_obj.name])
                    break
            if DEBUG:
                with open(os.path.join(cur_dir, "empties_matched.json"), "w", encoding = "utf-8") as f:
                    json.dump(empties_matched, f, indent = 4, ensure_ascii = False)
            return empties_filtered


def link_assetspath(cur_dir):
    # Find .blend file (assume there are only two .blend files in directory)
    filepath = None
    for i in os.listdir(cur_dir):
        if i.endswith(".blend") and os.path.basename(bpy.data.filepath) != i:
            filepath = os.path.join(cur_dir, i)
            break
    return filepath


def link_assets(filepath):
    """Link assets from the .blend file to the current scene"""
    # Link assets
    objs, cols = [], []
    
    with bpy.data.libraries.load(filepath, assets_only=True) as (data_from, data_to):
        for attr in dir(data_to):
            if attr == "objects":
                data_to.objects += data_from.objects
#                objs += data_from.objects
            elif attr == "collections":
                data_to.collections += data_from.collections
#                cols += data_from.collections

    bpy.ops.outliner.orphans_purge(do_local_ids=True, do_linked_ids=True, do_recursive=True)
                
    for i in range(len(bpy.data.materials)):
        bpy.data.materials[i].name = convert_name(bpy.data.materials[i].name)
    
    for image in bpy.data.images:
        c = False
        if image.name[-4] == ".":
            c = True
            for j in range(3):
                if not image.name[-(j + 1)].isnumeric():
                    c = False
                    break
        if c:
            image.name = image.name[:-4]


# def get_or_create_collection(name):
#     """Get or create a collection with the given name"""
#     collection = bpy.data.collections.get(name)
#     if collection is None:
#         collection = bpy.data.collections.new(name)
#         bpy.context.scene.collection.children.link(collection)
#     return collection


def link_obj_to_targets_parent_collections(obj, target_obj):
    """Link the given object to all collections that the target object is linked to"""
    for col in target_obj.users_collection:
        if obj.name not in col.objects:
            col.objects.link(obj)


def instantiate(scene, name, type_, empty_name):
    """Inherit the empty's T/R/S and preserve the instance's own local offset."""
    empty = bpy.data.scenes["Empties"].objects.get(empty_name)
    if empty is None:
        return None

    if type_ == 0:
        # Single object
        src = bpy.data.objects.get(name)
        if src is None:
            return None
        new_obj = src.copy()
        if getattr(src, "data", None) is not None:
            new_obj.data = src.data
        # scene.collection.objects.link(new_obj)
        link_obj_to_targets_parent_collections(new_obj, empty)
        # local offset = current local (world==local before parenting)
        local_offset = new_obj.matrix_basis.copy()
    else:
        # Collection instance
        col = bpy.data.collections.get(name)
        if col is None:
            return None
        new_obj = bpy.data.objects.new(name, None)
        new_obj.instance_type = 'COLLECTION'
        new_obj.instance_collection = col
        # scene.collection.objects.link(new_obj)
        link_obj_to_targets_parent_collections(new_obj, empty)
        local_offset = new_obj.matrix_basis.copy()  # usually identity

    # Parent and keep the instance's local offset:
    new_obj.parent = empty
    new_obj.matrix_parent_inverse = empty.matrix_world.inverted()
    new_obj.matrix_world = empty.matrix_world @ local_offset

    new_obj.empty_display_size = 0
    bpy.context.view_layer.objects.active = new_obj
    return new_obj


def make_scene(assetblend):
    #if bpy.data.filepath == '':
    #    return

    t0_total = time.perf_counter()

    cur_dir = os.path.dirname(assetblend)

    print("\nImporting assets...")
    usd_files = [p for p in get_props(cur_dir) if is_mesh_layer(p)]
    props = get_prop_types(usd_files)

    # print("\nGetting mesh references...")
    mesh_links = get_mesh_links(cur_dir)

    print("\nImporting empties...")
    empties_array = import_empties(props, cur_dir, mesh_links)

    rescale_point_clouds(100)

    print("\nLinking assets...")
    link_assets(assetblend)

    # # empties_col = get_or_create_collection("Empties")
    # for obj in bpy.data.scenes["Empties"].objects:
    #     for parent_col in obj.users_collection:
    #         if parent_col.name != "Empties":
    #             parent_col.objects.unlink(obj)
    #     bpy.context.scene.collection.objects.link(obj)
    #     # empties_col.objects.link(obj)

    print("\nInstantiating assets...")
    # Instantiate assets
    for i, empty_prop in enumerate(empties_array):
        if "landscape" in empty_prop[2].lower():
            print(f"Object: {i + 1}/{len(empties_array)}")
            print(1000, empty_prop[2], empty_prop[0])
        instantiate(bpy.context.scene, empty_prop[0], empty_prop[1], empty_prop[2])

    # Divide light powers by 100,000 to make them match with the Unreal scene -- OLD
    # Diving by 31.41 matches the new export method to the old / 100,000
    divide_light_powers(31.41)

    # Rename scene
    bpy.data.scenes.remove(bpy.data.scenes["Scene"])
    bpy.context.scene.name = "Scene"

    bpy.ops.object.select_all(action='DESELECT')
    t1_total = time.perf_counter()
    print(f"\nTotal time: {round(t1_total - t0_total, 3)} seconds\n")
