import bpy
import json
import os

from pathlib import Path
from pxr import Usd, UsdGeom, Sdf


USD_LOAD_NONE = Usd.Stage.LoadNone

def convert_name(name):
    """Convert empty's name into asset name"""
    nn = name.split("_|_")[0]
    if "." in name:
        return nn.split(".")[0]
    return nn


def clear_parent_and_keep_transform(obj):
    """Unparent object and keep transform"""
    if obj.parent is None:
        return
    world_matrix = obj.matrix_world.copy()
    obj.parent = None
    obj.matrix_world = world_matrix


def get_mesh_links(cur_dir):
    DEBUG = False
    for filename in os.listdir(os.path.join(cur_dir)):
        if not filename.endswith(".usd"):
            continue
        usd_path = os.path.join(cur_dir, filename)

        stage = Usd.Stage.Open(usd_path)
        if not stage:
            raise RuntimeError(f"Failed to open USD stage at {usd_path}")
        output_json_path = os.path.join(cur_dir, "linked_meshes.json")
        mesh_links = {}

        for prim in stage.Traverse():
            if prim.GetTypeName() in ["Shader", "Material", "MaterialBinding"]:
                continue

            mesh_path = str(prim.GetPath())
            parent =  prim.GetParent()

            mesh_path_last_two = mesh_path.split("/")[-1]
            if parent and parent.GetTypeName():
                mesh_path_last_two = "/".join(mesh_path.split("/")[-2:])

            for prim_spec in prim.GetPrimStack():
                if prim_spec.referenceList:
                    for ref in prim_spec.referenceList.prependedItems:
                        resolved = Sdf.ComputeAssetPathRelativeToLayer(prim_spec.layer, ref.assetPath)
                        resolved_filename = ".".join(os.path.basename(resolved).split(".")[:-1])
                        mesh_links[mesh_path_last_two] = resolved_filename

                    # if prim_spec.payloadList:
                    #     for payload in prim_spec.payloadList.prependedItems:
                    #         resolved = Sdf.ComputeAssetPathRelativeToLayer(prim_spec.layer, payload.assetPath)
                    #         mesh_links[mesh_path]["payloads"].append(resolved)
        if DEBUG:
            # Save to JSON
            with open(output_json_path, 'w') as f:
                json.dump(mesh_links, f, indent=4)
        return mesh_links


def is_mesh_layer(path: str) -> bool:
    """
    True ⟺ the layer has at least one Mesh prim.
    Uses LoadNone to avoid pulling in heavy payloads.
    """
    try:
        stage = Usd.Stage.Open(path, load=Usd.Stage.LoadNone)
    except Exception as exc:
        print(f"[USD-open-error] {path}: {exc}")
        return False

    for prim in stage.Traverse():          # cheap metadata-only walk
        if prim.IsA(UsdGeom.Mesh):
            return True
    return False


def get_props(cur_dir: str) -> list[str]:
    """
    Recursively collect every .usd inside <cur_dir>/Assets/ … that
    actually carries geometry.  Returned list is lexicographically sorted
    for deterministic behaviour downstream.
    """
    assets_root = Path(cur_dir) / "Assets"
    if not assets_root.is_dir():
        print(f"[get_props] No Assets folder in {cur_dir}")
        return []

    usd_paths: list[str] = []
    for usd_file in assets_root.rglob("*.usd"):
        # Optional: skip hidden folders / files if you need that
        if usd_file.name.startswith("."):
            continue
        if is_mesh_layer(str(usd_file)):
            usd_paths.append(str(usd_file))

    usd_paths.sort()          # keep reproducible order
    return usd_paths


def get_parent_from_selected():
    """
    Get the parent root object from the selected objects,
    useful when importing files
    """
    root_obj = None
    largest_child_count = -1
    for obj in bpy.context.view_layer.objects.selected:
        child_count = len(obj.children_recursive)
        if child_count > largest_child_count:
            largest_child_count = child_count
            root_obj = obj
    return root_obj


def get_relevant_child_obj(obj):
    """
    Get the object that will be used for instantiation, or for marking as asset
    """
    if len(obj.children) == 0:
        return obj
    elif len(obj.children) == 1:
        return obj.children[0]
    else:
        for child in obj.children_recursive:
            if child.type == "MESH" or child.type == "LIGHT" or child.type == "CAMERA":
                return child
    return None


def imported_kind(root_obj: bpy.types.Object) -> int:
    """
    Return 0 → single-object   (one mesh, no extras)
           1 → collection      (anything else)
    """
    objs = [root_obj, *root_obj.children_recursive]
    meshes     = [o for o in objs if o.type == 'MESH']
    non_meshes = [o for o in objs if o.type != 'MESH']
    return 0 if len(meshes) == 1 and len(non_meshes) == 0 else 1


def get_prop_types(props):
    """Get array with [path, type, name] of assets"""

    arr, objs = [], []
    
    for prop_path in props:
        bpy.ops.object.select_all(action='DESELECT')
        bpy.ops.wm.usd_import(
            filepath=prop_path,
            create_collection=False
        )

        root_obj = get_parent_from_selected()
        if root_obj is None:
            # print(f"Could not find root object for {prop_path}")
            continue

        # prop_type = 1 if len(root_obj.children) == 1 else 0  # 0 - Object; 1 - Collection
        prop_type = imported_kind(root_obj)
        if prop_type == 1:
            name_ = root_obj.name
        else:
            name_ = None

        # arr.append([prop_path, prop_type, name_ or ".".join(os.path.basename(prop_path).split(".")[:-1])])
        arr.append([prop_path, prop_type, ".".join(os.path.basename(prop_path).split(".")[:-1])])

    bpy.ops.object.select_all(action='SELECT')
    bpy.ops.object.delete()
    return arr
