import re
import bpy
import os
import json
import math
from .daily import Daily
import mathutils
from . import CameraPlane
from .argutil import ImportArguments,SceneType,RunScope
from .Composite import CompositeKey,ScreenResolution
from . import imageloading
from . import icons
from .utiloperators import gatherscenelocs
from . import version
from math import degrees, radians, tan
from mathutils import Vector

from .exportbase import ExportBase,ExportCamera,CameraFrame,readserialized

LIDARWIDTH=256
LIDARHEIGHT=144
SLIGHTADJUSTMENTFRAMES=0
NANO100TICKS=10000000.0
CENTERINTRINSICS=True
RENDERSCENE="Render"

class FrameIntermediate:
    def __init__(self,w,h):
        self._width = w
        self._height = h
        
    def parse(self,frame,sensorwidth,focallength):
        self.tag = frame["tag"]

        ad = frame["pose"]
        self._posv = mathutils.Vector((float(ad[0]), float(ad[1]), float(ad[2])))
        ad = frame["forward"]
        self._forv = mathutils.Vector((float(ad[0]), float(ad[1]), float(ad[2])))
        ad = frame["up"]
        self._upv = mathutils.Vector((float(ad[0]), float(ad[1]), float(ad[2])))

        self._cx = self._width / 2
        self._cy = self._height / 2

        if CENTERINTRINSICS:
            if "cx" in frame:
                self._cx = float(frame["cx"])
            if "cy" in frame:
                self._cy = float(frame["cy"])

        self._sensorwidth = sensorwidth        
        self._focallength = focallength        

        #mmperp = self._sensorwidth / self._width

        self._cameraPlaneX = self._cx - self._width / 2
        self._cameraPlaneY = self._cy - self._height / 2    

        self._xfov = float(frame["xFovDegrees"])

        personDepth = 0.0
        if "personDepth" in frame:
            personDepth = float(frame["personDepth"])

        self._personDepth = personDepth

            


class AFTERBURNER_PT_Panel(bpy.types.Panel):
    bl_idname = "AFTERBURNER_PT_Panel"
    bl_label = "Autoshot"
    bl_space_type = 'VIEW_3D'
    bl_region_type = 'UI'
    bl_category = "Autoshot"
    bl_context = "objectmode"    
    
        #from . import addon_properties
        #addon_version = addon_properties.addon_version

    def draw(self, context):
        layout = self.layout

        scn = context.scene

        row = layout.row()  
        row.label(text=f"v{version.addon_stringversion}")              

        #row.prop(scn.AFTERBURNER,"trackingpath", text="Tracking JSON")
        row = layout.row()                
        #layout.separator()
        row.operator(ViewportRenderOperator.bl_idname, text="Fast Viewport Video", icon="RENDER_ANIMATION")

        row = layout.row()                
        row.operator(StandardRenderOperator.bl_idname, text="Standard Render Settings", icon="RENDER_ANIMATION")

            # Box gives the pretty outline grouping
        #box = layout.box()
        #box.label(text="Camera Plane")

        #row = box.row()                
        #row.prop(scn.AFTERBURNER,"depth_amount", text="Explicit Depth")

        #box = layout.box()
        #box.label(text="Tracking")

        #row = box.row()                
        #row.prop(scn.AFTERBURNER,"fps", text="Frames Per Second")
        #row = box.row()                
        #row.prop(scn.AFTERBURNER,"shift_amount", text="Tracking Shift")


        #row = layout.row()                
        #row.prop(scn.AFTERBURNER,"secondcamera_boolean", text="Second Camera")

        #row = layout.row()                
        #row.prop(scn.AFTERBURNER,"texturepath2", text="Texture Dir 2")

        #row = layout.row()                
        #row.prop(scn.AFTERBURNER,"fps2", text="Frames Per Second 2")

       # print the path to the console
        #print ("in draw ",scn.AFTERBURNER.texturepath)

        layout.separator()

        op = layout.operator(OpenUrlOperator.bl_idname, text="Tutorials", icon="URL")
        url = "https://www.lightcraft.pro/tutorials"
        OpenUrlOperator.configure(op, url)

        op = layout.operator(OpenUrlOperator.bl_idname, text="Community", icon="URL")
        url = "https://forums.lightcraft.pro"
        OpenUrlOperator.configure(op, url)

        #op = layout.operator(OpenUrlOperator.bl_idname, text="Discord", icon_value=icons.get('discord'))
        #url = "https://discord.gg/RkQfcCNeFm"
        #OpenUrlOperator.configure(op, url)


        #row = layout.row()                
        #row.prop(scn.AFTERBURNER,"scrub_boolean", text="Automatic Timeline Shot Switching")

        #layout.separator()

        #col.label(text="Debug Stuff:")
        #row = layout.row()  
        #row.operator(ViewportOperator.bl_idname, text="Fast Viewport Render", icon="RENDER_ANIMATION")
        #row = layout.row()  
        #row.prop(scn.AFTERBURNER,"fastdebug_boolean", text="PNG Render - for testing")

        layout.separator()
        #row = layout.row()                
        #row.label(text="Optional for single files")
        #row = layout.row()                
        #row.prop(scn.AFTERBURNER,"texturepath", text="Texture/Lidar Parent")


class OpenUrlOperator(bpy.types.Operator):
    '''Open a URL in the browser'''
    bl_idname = 'afterburner.url_open'
    bl_label = 'Open URL'

    url: bpy.props.StringProperty(name='URL', options={'HIDDEN', 'SKIP_SAVE'})

    def execute(self, context: bpy.types.Context) -> set:
        bpy.ops.wm.url_open(url=self.url)
        return {'FINISHED'}

    @staticmethod
    def configure(op: bpy.types.OperatorProperties, url: str):
        op.url = url


class DailyOperator(bpy.types.Operator):
    """Import and get ready to render a daily"""
    bl_idname = "afterburner.daily"
    bl_label = "Daily Operator"

    def execute(self, context):
        #myprops = context.scene.AFTERBURNER

        r = bpy.ops.afterburner.import_tracking() # Call the execute()
        #code = r.pop()
        #if code == 'FINISHED': For when import 2 trackes
        #    if myprops.cinetrackingpath != "":
        #        myprops.composite_boolean = False # Already have scene imported
        #        myprops.trackingpath = myprops.cinetrackingpath
        #        if namesuffix == "_cine":
        #            myprops.campath = myprops.cinecampath
        #            myprops.depthpath = ""
        #        myprops.namesuffix = namesuffix
        #        r = bpy.ops.afterburner.import_tracking() # Call the execute()
        #        code = r.pop()

        #if code == 'FINISHED':
        #    d.finish()
        return r

class ViewportRenderOperator(bpy.types.Operator):
    """Set files and call Viewport Render Animation"""
    bl_idname = "afterburner.viewportrender"
    bl_label = "Viewport Render Operator"

    def execute(self, context):
        print ("ViewportRenderOperator")
        ia = ImportArguments()
        myprop = context.scene.AFTERBURNER
        ia.cameraname = myprop.cameraname
        print (f"ia.cameraname {ia.cameraname}")

        d = Daily(ia)
        d.setup(False)
        d.fastrender()
        return {'FINISHED'}

class StandardRenderOperator(bpy.types.Operator):
    """Set files for single frame renders"""
    bl_idname = "afterburner.standardrender"
    bl_label = "Standard Render Settings Operator"

    def execute(self, context):
        print ("StandardRenderOperator")
        ia = ImportArguments()
        myprop = context.scene.AFTERBURNER
        ia.cameraname = myprop.cameraname
        d = Daily(ia)
        d.normalrendersetup()
        return {'FINISHED'}


from bpy_extras.io_utils import ImportHelper
from bpy.props import StringProperty, BoolProperty, EnumProperty, FloatProperty
from bpy.types import IMAGE_HT_tool_header, Operator

    
def FocalLengthmm(sensorwidth,xfov):
    fl = (sensorwidth / 2) / math.tan(math.radians (xfov / 2))
    return fl


class ImportOperator(Operator, ImportHelper):
    """Import Tracking File or list"""
    _instance = None
    bl_idname = "afterburner.import_tracking" 
    bl_label = "Autoshot (.json)"
    # ImportHelper mixin class uses this

    # By default all collections get the enable state of the current active view layer
    # Find exclude for any collections we created for camera shots that are not this shot
    def include_exclude_collection(self, layer_collection: bpy.types.Collection, collection_include: bpy.types.Collection,
                    bookkeeping: dict):
        for collection in layer_collection.children:
            found = False
            for shotname,d in bookkeeping.items():
                if d['col'] == collection.collection:
                    found = True
                    break
            
            if found:
                if collection.collection != collection_include:
                    collection.exclude = True
            #else:
                #srcfound, collection.exclude = self.currentviewlayer_collection_current_exclude(collection.collection)

            # recurse through children of this collection
            #self.include_exclude_collection(collection, collection_include, bookkeeping)

    # By default all collections get the enable state of the current active view layer
    # Find exclude for any collections we created for camera shots that are not this shot
    # Do this for this new view layer
    def layer_include_exclude_collection(self, view_layer: bpy.types.ViewLayer, collection_include: bpy.types.Collection,
                    bookkeeping: dict):
            self.include_exclude_collection(view_layer.layer_collection, collection_include, bookkeeping)

    def collection_current_exclude(self, srccollection: bpy.types.Collection, collection: bpy.types.Collection):
        for layer_collection in srccollection.children:
            if layer_collection.collection == collection:
                return True, layer_collection.exclude
            found,val = self.collection_current_exclude(layer_collection,collection)
            if found:
                return True,val
        print("Could not find ",collection.name)
        return False, False

    def currentviewlayer_collection_current_exclude(self, collection: bpy.types.Collection):
        return self.collection_current_exclude(bpy.context.view_layer.layer_collection, collection)
            
    #Recursivly transverse layer_collection for a particular name
    def recurLayerCollection(self,layerColl, collName):
        found = None
        if (layerColl.name == collName):
            return layerColl
        for layer in layerColl.children:
            found = recurLayerCollection(layer, collName)
            if found:
                return found

    def removeviewlayerifexists(self,name):
        for v in bpy.context.scene.view_layers:
            if v.name == name:
                print(f"Deleting viewlayer {v.name}")
                bpy.context.window.view_layer = v
                bpy.ops.scene.view_layer_remove()
                bpy.context.window.view_layer = bpy.context.scene.view_layers[0]
                return

    def setscenecollectionactive(self):
        #Change the Active LayerCollection to 'Scene Collection'
        layer_collection = bpy.context.view_layer.layer_collection
        layerColl = self.recurLayerCollection(layer_collection, 'Scene Collection')
        bpy.context.view_layer.active_layer_collection = layerColl

    def set_camerabackground(self, context, ia:ImportArguments,camobj):
        (path,isdir) = imageloading.TextureFind(None,None,ia.shotname)
        if path != None:
            img = bpy.data.images.load(path)
            camobj.data.show_background_images = True
            bg = camobj.data.background_images.new()
            bg.image = img

    def recursecatcher(self, ob):
        if hasattr(ob,"is_shadow_catcher"):
            print(f"is_shadow_catcher set for {ob.name}")
            ob.is_shadow_catcher = True
        else:
            print(f"{ob.name} doesn't have is_shadow_catcher")

        if ob.type == 'MESH':
            print("Change to WIRE")
            ob.display_type = 'WIRE'


        for o in ob.children:
            self.recursecatcher(o)    

    def import_externalscan(self, ia:ImportArguments, parent):
        if ia.exportbase.externalscanpath == "":
            return


        current_collection = bpy.context.view_layer.active_layer_collection
        collection = bpy.data.collections.new("External Scan Collection")
        bpy.context.scene.collection.children.link(collection)

        # Put the USD contents in this collection
        layer_collection = bpy.context.view_layer.layer_collection.children[collection.name]

        bpy.context.view_layer.active_layer_collection = layer_collection

        # Import the OBJ file
        try:
            bpy.ops.wm.obj_import(filepath=ia.exportbase.externalscanpath)
        except:
            print("Failed import obj file")
            return

        # bpy.data.objects["Plane"].is_shadow_catcher
        # Create containing object
        ob = None
        for o in collection.objects:
            if o.parent == None:
                ob = o
                break

        if ob != None:
            name = ob.name
            print(f"top object name in external scan OBJ is {name} parent {ob.parent}")
            ob.parent = parent


            # Join all the Meshes
            bpy.ops.object.select_all(action='DESELECT')

            for o in collection.objects:
                if o.type == 'MESH':
                    o.select_set(True)
                    #Makes one active
                    bpy.context.view_layer.objects.active = o

            # Get the currently selected objects
            selected_objects = bpy.context.selected_objects
            # Check if there are at least two selected objects
            if len(selected_objects) >= 2:
                bpy.ops.object.join()

            self.recursecatcher(ob)


        # Hide the collection in the viewport and disable it for rendering by default.
        layer_collection.hide_viewport = True
        layer_collection.exclude = True
        bpy.context.view_layer.active_layer_collection = current_collection


    def import_scan(self, ia:ImportArguments, parent):
        if ia.exportbase.scanpath == "":
            return


        current_collection = bpy.context.view_layer.active_layer_collection
        collection = bpy.data.collections.new("Scan Collection")
        bpy.context.scene.collection.children.link(collection)

        # Put the USD contents in this collection
        layer_collection = bpy.context.view_layer.layer_collection.children[collection.name]
        bpy.context.view_layer.active_layer_collection = layer_collection

        # I could not get create_collection=True to work. blender bug?
        #bpy.ops.wm.usd_import(filepath=ia.exportbase.scanpath,relative_path=False,import_materials=True,create_collection=False,import_usd_preview =True,
        #                    import_all_materials=True,import_meshes=True)
        bpy.ops.wm.usd_import(filepath=ia.exportbase.scanpath)

        # bpy.data.objects["Plane"].is_shadow_catcher
        # Create containing object
        ob = None
        for o in collection.objects:
            if o.parent == None:
                ob = o
                break

        if ob != None:
            name = ob.name
            print(f"top object name in USD is {name} parent {ob.parent}")
            ob.parent = parent


            # Join all the Meshes
            bpy.ops.object.select_all(action='DESELECT')

            for o in collection.objects:
                if o.type == 'MESH':
                    o.select_set(True)
                    #Makes one active
                    bpy.context.view_layer.objects.active = o

            # Get the currently selected objects
            selected_objects = bpy.context.selected_objects
            # Check if there are at least two selected objects
            if len(selected_objects) >= 2:
                bpy.ops.object.join()

            # Put a non-empty above the Joined mesh
            #me = bpy.data.meshes.new("Holder")
            #newob = bpy.data.objects.new("Holder", me)
            #ob.parent = newob
            #collection.objects.link(newob)

            self.recursecatcher(ob)

        # Hide the collection in the viewport and disable it for rendering by default.
        layer_collection.hide_viewport = True
        layer_collection.exclude = True

        bpy.context.view_layer.active_layer_collection = current_collection



    def getMatrix(self, fi:FrameIntermediate):
        upv = mathutils.Vector((fi._upv[0], fi._upv[1], fi._upv[2]))
        forv = mathutils.Vector((fi._forv[0], fi._forv[1], fi._forv[2]))
        #eulxyz = mathutils.Vector((fi._eulxyz[0], fi._eulxyz[1], fi._eulxyz[2]))

        #print("frame count ",framecount)
        # Blender is Z up model.
        # Blender camera looks down -Z axis
        # ArKit is Y up model
        # Camera looks down -Z axis

        # Straight up
        #upv = mathutils.Vector((0,1,0))
        #forv = mathutils.Vector((0,0,1))

        # Tilt Forward
        #upv = mathutils.Vector((0,0.707,-.707))
        #forv = mathutils.Vector((0,0.707,0.707))

        # Tilt Down
        #upv = mathutils.Vector((0, 0, -1))
        #forv = mathutils.Vector((0,1.0,0.0))

        unitchange = 1.0  # Meters to centimeters
        posv = mathutils.Vector((fi._posv[0] * unitchange, fi._posv[1] * unitchange, fi._posv[2] * unitchange))
        #posv = mathutils.Vector((0,2,-3))

        ritv = upv.cross(forv)
        ritv.normalize()
        #print(f"setPrim framecount {framecount}")
        #print(f"ritv {ritv}")
        #print(f"crossv {fi._crossv}")
        
        # Swapping Z and -Y to get from ARKit to Blender Coordinates
        mat = mathutils.Matrix( 
                ((ritv[0], upv[0], forv[0], posv[0]),
                (-ritv[2],  -upv[2], -forv[2], -posv[2]),
                (ritv[1],  upv[1], forv[1], posv[1]),
                (0,    0,       0,    1))
                )
        #print("mat \n",mat)     
        return mat  

    def setPrimCamera(self, cf:CameraFrame,cam:ExportCamera,cameraObject,cameraPlane,persondepth, framecount):

                # Create a Blender Matrix for the rotation part
        rotation_matrix = mathutils.Matrix(cf.rotmat)

        # Create a 4x4 matrix from the 3x3 rotation matrix
        transform_mat = rotation_matrix.to_4x4()

        # Set the translation part of the 4x4 matrix
        transform_mat.translation = mathutils.Vector(cf.pose)

        #fi._xfov = 64.0 # hard coding test
        CurrentFocalLength = FocalLengthmm(cam.sensorwidth,cf.xFovDegrees)
        #s = 30 * math.tan(degreeradians * (fi._xfov / 2)) * 2
        #print("s ",s)
        #cameraObject.data.lens_unit = 'MILLIMETERS'
        #print("CurrentFocalLength",CurrentFocalLength)
        #CurrentFocalLength = 30.0

        cameraObject.data.lens = CurrentFocalLength
        if cam.focallength != 0: # User interface nmodification of focal of camera
            cameraObject.data.lens = cam.focallength
   
        if CENTERINTRINSICS and cam.width != 0:
            cameraObject.data.shift_x = -(cf.cx / cam.width - 0.5)
            cameraObject.data.shift_y = (cf.cy / cam.height - 0.5) * (cam.height/cam.width)
   
        #print(f"shift {cameraObject.data.shift_x} {cameraObject.data.shift_y}")
        cameraObject.data.sensor_width = cam.sensorwidth
        cameraObject.data.sensor_height = cam.sensorwidth * (cam.height/cam.width)
        cameraObject.data.sensor_fit = 'HORIZONTAL'

        cameraObject.matrix_local = transform_mat
        cameraObject.data.keyframe_insert("lens", frame=framecount)
        cameraObject.data.keyframe_insert("shift_x", frame=framecount)
        cameraObject.data.keyframe_insert("shift_y", frame=framecount)
        cameraObject.keyframe_insert("location", frame=framecount)
        cameraObject.keyframe_insert("rotation_euler", frame=framecount)

        if cameraPlane != None:
            if persondepth and cf.depth != 0.0:
                self._lastdepth = cf.depth

            #cameraPlane.location.z = -self._lastdepth
            #cameraPlane.location.x = fi._cameraPlaneX * self._lastdepth / (CurrentFocalLength * 1000)
            #CurrentFocalLengthY = CurrentFocalLength * fi._height / fi._width
            #cameraPlane.location.y = fi._cameraPlaneY * self._lastdepth / (CurrentFocalLengthY * 1000)
            cameraPlane.location.z = 0
            if CENTERINTRINSICS:
                cameraPlaneX = cf.cx - cam.width / 2
                cameraPlaneY = cf.cy - cam.height / 2   
            else: 
                cameraPlaneX = 0
                cameraPlaneY = 0   

            cameraPlane.location.x = cameraPlaneX  / (CurrentFocalLength * 1000)
            CurrentFocalLengthY = CurrentFocalLength * cam.height / cam.width
            cameraPlane.location.y = cameraPlaneY  / (CurrentFocalLengthY * 1000)
            cameraPlane.keyframe_insert("location", frame=framecount)
            CameraPlane.setplanescale(cameraPlane,math.radians(cf.xFovDegrees), self._lastdepth)
            cameraPlane.keyframe_insert("scale", frame=framecount)

    def _findplane(self, co):
        for ob in co.children:
            if CameraPlane.ourplanename(ob.name):
                return ob
        return None
               
    def import_file_tracking(self, context, ia:ImportArguments):

        cam,plane = self.import_filelow_tracking(context, ia)
        if cam != None:
            cam.select_set(True)
            context.scene.camera = cam

            context.scene.frame_set(ia.exportbase.timelinestartframe)
            context.scene.frame_start = ia.exportbase.timelinestartframe
            context.scene.frame_end = context.scene.frame_start + (ia.exportbase.endframe-ia.exportbase.startframe) 
            #print(f"set to {context.scene.frame_end} s {context.scene.frame_start} ef {ia.end_frame} sf {ia.start_frame}")
            return True
        return False

    def do_copysettings(self, srcstruct, dststruct):
        for prop in srcstruct.bl_rna.properties:
            if prop.identifier in {'rna_type'} or prop.is_readonly:
                continue
            try:
                # Get the property value from the source
                value = getattr(srcstruct, prop.identifier)
            except Exception as e:
                print(f"Failed to read property '{prop.identifier}': {e}")
                continue
            try:
                # Optionally, check if the destination even has the property
                if hasattr(dststruct, prop.identifier):
                    setattr(dststruct, prop.identifier, value)
                else:
                    print(f"Destination has no property '{prop.identifier}', skipping.")
            except Exception as e:
                print(f"Failed copying property '{prop.identifier}': {e}")

    def _promote_locators_to_local(self, context, topscene):
        """
        Finds local objects that match specific locator prefixes, de-duplicates them,
        and promotes the unique ones to a permanent, dedicated collection.
        """
        LOCATOR_PREFIXES = ('sceneloc_', 'splatloc_', 'imageloc_')
        TARGET_COLLECTION_NAME = "SceneLocators"

        # 1. Get or create the target collection
        locator_collection = bpy.data.collections.get(TARGET_COLLECTION_NAME)
        if not locator_collection:
            locator_collection = bpy.data.collections.new(TARGET_COLLECTION_NAME)
            topscene.collection.children.link(locator_collection)
            print(f"Created collection: '{TARGET_COLLECTION_NAME}'")

        # 2. Find all potential locators by name
        potential_locators = {obj.name: obj for obj in bpy.data.objects if obj.library is None and obj.name.startswith(LOCATOR_PREFIXES)}
        
        # 3. Identify and remove duplicates
        objects_to_remove = []
        for name, obj in potential_locators.items():
            match = re.match(r"^(.*)\.\d{3}$", name)
            if match and match.group(1) in potential_locators:
                objects_to_remove.append(obj)

        if objects_to_remove:
            print(f"Found and removing {len(objects_to_remove)} duplicate locators.")
            for obj in objects_to_remove:
                print(f"  - Removing duplicate locator '{obj.name}'")
                if obj.name in potential_locators:
                    del potential_locators[obj.name]
                bpy.data.objects.remove(obj, do_unlink=True)

        # 4. Promote the unique locators
        promoted_locators_names = []
        for name, obj in potential_locators.items():
            if obj.name not in locator_collection.objects:
                print(f"Promoting object '{obj.name}' to a permanent local locator.")
                # Unlink from all other collections to ensure it's only in ours.
                for coll in obj.users_collection:
                    coll.objects.unlink(obj)
                locator_collection.objects.link(obj)
                promoted_locators_names.append(obj.name)

        if promoted_locators_names:
            print(f"Successfully promoted: {promoted_locators_names}")
        else:
            print(f"No locators with prefixes {LOCATOR_PREFIXES} found to promote.")

    def linkexternalscene(self, context, sceneblendpath: str):
        """
        Links assets from an external .blend file, intelligently merging the default "Collection"
        to avoid name clashes, and handling settings. This version is optimized to load data only once.
        """
        if not os.path.exists(sceneblendpath):
            print(f"Error: Scene blend file not found at '{sceneblendpath}'")
            return

        SETTINGS_SCENE_NAME = "Scene"
        blend_path_str = str(sceneblendpath)

        try:
            topscene = bpy.data.scenes[RENDERSCENE]
        except KeyError:
            print(f"Error: Could not find the main render scene named '{RENDERSCENE}'")
            return

        # --- PART 1: Link all data from the external file in a single operation ---
        print(f"--- Linking all data from '{os.path.basename(blend_path_str)}' ---")
        with bpy.data.libraries.load(blend_path_str, link=True) as (data_from, data_to):
            data_to.collections = [name for name in data_from.collections if not name.startswith('.')]
            data_to.objects = [name for name in data_from.objects]

        # --- PART 2: Intelligently merge the linked "Collection" into the local one ---
        loaded_lib = bpy.data.libraries.get(os.path.basename(blend_path_str))
        if not loaded_lib:
            print(f"Error: Could not get library handle for '{os.path.basename(blend_path_str)}'")
            return

        # Find the linked collection that came from the source file's "Collection"
        source_collection_linked = next((c for c in bpy.data.collections if c.library == loaded_lib and c.name == "Collection"), None)
        
        # Find the pre-existing, local "Collection"
        local_collection = next((c for c in bpy.data.collections if c.library is None and c.name == "Collection"), None)

        if source_collection_linked and local_collection:
            print("Merging linked 'Collection' into existing local 'Collection'.")
            
            # Merge child objects
            for obj in source_collection_linked.objects:
                if obj.name not in local_collection.objects:
                    local_collection.objects.link(obj)
            
            # Merge child collections
            for child_coll in source_collection_linked.children:
                if child_coll.name not in local_collection.children:
                    local_collection.children.link(child_coll)

        # --- PART 3: Link the top-level collections to the scene ---
        lib_collections = {c for c in bpy.data.collections if c.library == loaded_lib}
        if lib_collections:
            child_collections = {child for coll in lib_collections for child in coll.children if child in lib_collections}
            top_level_collections = lib_collections - child_collections
            print(f"lib_collections: {[c.name for c in lib_collections]}")
            print(f"child_collections: {[c.name for c in child_collections]}")

            if top_level_collections:
                print(f"Linking top-level collections: {[c.name for c in top_level_collections]}")
                for coll in top_level_collections:
                    # CRUCIAL: Skip linking the source "Collection" as we've merged its contents.
                    if coll == source_collection_linked:
                        print(f"  - Skipping link of source 'Collection' as its content has been merged.")
                        continue
                    
                    if coll.name not in topscene.collection.children:
                        topscene.collection.children.link(coll)
                        print(f"  - Linked collection '{coll.name}' and its hierarchy to the scene.")
            else:
                print("Warning: No top-level collections found to link.")

        # --- PART 4: Link any uncollected objects ---
        for obj in bpy.data.objects:
            if obj.library == loaded_lib and not obj.users_collection and not obj.name.startswith("sceneloc_"):
                print(f"  - Linking uncollected object '{obj.name}' to the root scene collection.")
                topscene.collection.objects.link(obj)

        # --- PART 5: Handle World and Render Settings ---
        print("--- Processing World and Render Settings ---")
        # Get available data-block names and link the world in a more robust way.
        with bpy.data.libraries.load(blend_path_str, link=False) as (data_from, _):
            available_scene_names = data_from.scenes
            available_world_names = data_from.worlds

        if available_world_names:
            with bpy.data.libraries.load(blend_path_str, link=True) as (data_from, data_to):
                data_to.worlds = [available_world_names[0]]

            if data_to.worlds:
                topscene.world = data_to.worlds[0]
                print(f"Linked and assigned world: '{topscene.world.name}'")
            else:
                print(f"Warning: Failed to link world '{available_world_names[0]}'.")
        else:
            print("No world found in source file to link.")

        if SETTINGS_SCENE_NAME in available_scene_names:
            with bpy.data.libraries.load(blend_path_str, link=False) as (data_from, data_to):
                data_to.scenes = [SETTINGS_SCENE_NAME]
            source_scene = bpy.data.scenes.get(SETTINGS_SCENE_NAME)
            if source_scene:
                print("Copying render and cycles settings...")
                try:
                    self.do_copysettings(source_scene.render, topscene.render)
                    self.do_copysettings(source_scene.cycles, topscene.cycles)
                except AttributeError as e:
                    # This can happen when running headlessly, as some property updates
                    # try to redraw a UI area which doesn't exist. It's safe to ignore.
                    print(f"Ignoring a minor UI update error during settings copy: {e}")
                print("Done copying settings.")
                # IMPORTANT: We remove the scene, but the objects it contained now exist
                # in memory as "ghosts".
                bpy.data.scenes.remove(source_scene)

        # --- PART 6: Promote locators from the temporary settings scene ---
        self._promote_locators_to_local(context, topscene)


    def appendexternalscene(self, context, sceneblendpath: str):
        """
        Appends assets from an external .blend file, preserving collection hierarchy.
        Collections and objects are handled first to prevent duplication from settings import.
        """
        if not os.path.exists(sceneblendpath):
            print(f"Error: Scene blend file not found at '{sceneblendpath}'")
            return

        SETTINGS_SCENE_NAME = "Scene"
        blend_path_str = str(sceneblendpath)

        try:
            topscene = bpy.data.scenes[RENDERSCENE]
        except KeyError:
            print(f"Error: Could not get render scene '{RENDERSCENE}': {e}")
            return

        # --- PART 1: Append all objects and collections first to avoid duplication ---
        print(f"--- Appending collections from '{os.path.basename(blend_path_str)}' ---")
        appended_objects = []
        with bpy.data.libraries.load(blend_path_str, link=False) as (data_from, data_to):
            # Append all user-facing collections. This brings their objects with them.
            data_to.collections = [name for name in data_from.collections if not name.startswith('.')]
            # Append all objects. Blender is smart and will only append objects that
            # don't already exist, catching any objects not in a collection.
            data_to.objects = [name for name in data_from.objects]
            appended_objects = data_to.objects

        appended_collections = set(data_to.collections)
        if appended_collections:
            child_collections = set()
            for coll in appended_collections:
                for child in coll.children:
                    if child in appended_collections:
                        child_collections.add(child)
            top_level_collections = appended_collections - child_collections
            if top_level_collections:
                print(f"Appending top-level collections: {[c.name for c in top_level_collections]}")
                for coll in top_level_collections:
                    if coll.name not in topscene.collection.children:
                        topscene.collection.children.link(coll)
                        print(f"  - Appended collection '{coll.name}' and its hierarchy to the scene.")
            else:
                print("Warning: No top-level collections found to append.")

        # Link any appended objects that are not in a collection to the main scene.
        for obj in appended_objects:
            # An object is "uncollected" if it has no users in any collection.
            if obj and not obj.users_collection:
                if obj.name not in topscene.collection.objects:
                    print(f"  - Linking uncollected object '{obj.name}' to the root scene collection.")
                    topscene.collection.objects.link(obj)

        # Post-process to merge the common "Collection" name collision.
        coll_dupe = bpy.data.collections.get("Collection.001")
        coll_orig = bpy.data.collections.get("Collection")

        if coll_dupe and coll_orig:
            print("Merging appended 'Collection.001' into existing 'Collection'.")
            
            # Move all objects from the duplicate to the original
            for obj in list(coll_dupe.objects):
                if obj.name not in coll_orig.objects:
                    coll_orig.objects.link(obj)
                coll_dupe.objects.unlink(obj)

            # Move all child collections from the duplicate to the original
            for child in list(coll_dupe.children):
                if child.name not in coll_orig.children:
                    coll_orig.children.link(child)
                coll_dupe.children.unlink(child)
            
            bpy.data.collections.remove(coll_dupe)

        # --- PART 2: Handle World and Render Settings ---
        print("--- Processing World and Render Settings ---")
        with bpy.data.libraries.load(sceneblendpath, link=False) as (data_from, data_to):
            if data_from.worlds:
                data_to.worlds = [data_from.worlds[0]]
            if SETTINGS_SCENE_NAME in data_from.scenes:
                data_to.scenes = [SETTINGS_SCENE_NAME]

        if data_to.worlds:
            topscene.world = data_to.worlds[0]
            print(f"Appended and assigned world: '{topscene.world.name}'")
        else:
            print("No world found in source file to append.")

        source_scene = bpy.data.scenes.get(SETTINGS_SCENE_NAME)
        if source_scene:
            print("Copying render and cycles settings...")
            try:
                self.do_copysettings(source_scene.render, topscene.render)
                self.do_copysettings(source_scene.cycles, topscene.cycles)
            except AttributeError as e:
                print(f"Ignoring a minor UI update error during settings copy: {e}")
            print("Done copying settings.")
            bpy.data.scenes.remove(source_scene)

        # --- PART 3: Promote locators from the temporary settings scene ---
        self._promote_locators_to_local(context, topscene)



    def addscenelocused(self,ia:ImportArguments):
        scenelocused = ia.exportbase.scenelocused

        rotation_matrix = mathutils.Matrix(scenelocused.rotmat)

        # Create a 4x4 matrix from the 3x3 rotation matrix
        transform_mat = rotation_matrix.to_4x4()

        # Set the translation part of the 4x4 matrix
        transform_mat.translation = mathutils.Vector(scenelocused.pose)

        bpy.ops.object.empty_add(type='ARROWS')

        empty_object = bpy.context.active_object
        empty_object.name = "scenelocused"
        empty_object.matrix_local = transform_mat

        #bpy.ops.object.camera_add(enter_editmode=False)
        #empty_object = bpy.context.active_object
        #empty_object.name = "Scenelocused"
        #empty_object.matrix_local = transform_mat



    def adjust_timeline_view(self, context):
        """
        Finds the timeline editor and adjusts its view to fit all keyframes
        using a context override. It forces a dependency graph update before
        the operator call to ensure the timeline data is current.
        This is the most robust method available given API limitations.
        """
        # Force a full update of the dependency graph. This is crucial.
        depsgraph = context.evaluated_depsgraph_get()
        depsgraph.update()

        for area in context.screen.areas:
            if area.type == 'DOPESHEET_EDITOR':
                space = next((s for s in area.spaces if s.type == 'DOPESHEET_EDITOR' and s.mode == 'TIMELINE'), None)

                if space:
                    override_context = context.copy()
                    override_context['area'] = area
                    window_region = next((r for r in area.regions if r.type == 'WINDOW'), None)
                    if not window_region:
                        continue
                    override_context['region'] = window_region

                    try:
                        with context.temp_override(**override_context):
                            bpy.ops.action.view_all()
                        print(f"Adjusted timeline view to fit all keyframes.")
                        return 
                    except RuntimeError as e:
                        print(f"Could not adjust timeline view: {e}")


        # In Autoimport.py -> ImportOperator class



    def import_filelow_tracking(self, context, ia:ImportArguments):
        fps = ia.exportbase.fps

        if ia.exportbase.depth_amount != 0:
            self._lastdepth = ia.exportbase.depth_amount
        else:
            self._lastdepth = 2 # Default depth for camera plane z axis
        C = context
        O = bpy.ops

        if ia.scenetype == SceneType.APPEND_SCENE:
            self.appendexternalscene(context,ia.sceneblendpath)
        elif ia.scenetype == SceneType.LINK_SCENE:
            self.linkexternalscene(context,ia.sceneblendpath)


        ### Global Scene Settings
        if fps == int(fps):
            C.scene.render.fps = int(fps)
            C.scene.render.fps_base = 1.0
        else:
            C.scene.render.fps = int(fps*100 + 0.5)
            C.scene.render.fps_base = 100.0


        # Add the Camera
        if context.active_object != None:
            bpy.ops.object.mode_set(mode='OBJECT')
        O.object.camera_add(enter_editmode=False)
        cameraObject = C.active_object

        cameraname = ia.cameraname
        if cameraname == None:
            cameraname = "JetsetCamera"

        cameraObject.name =  cameraname + ia.exportbase.namesuffix

        scenelocname = ia.exportbase.sceneloc
        scenelocrotation = ia.exportbase.scenelocrotation
        print(f"sceneloc is {scenelocname}")
         
        if scenelocname != "origin":
            ob = bpy.data.objects.get(scenelocname)
            if ob != None:
                cameraObject.parent = ob
                print(f"Setting camera parent {ob.name}")
            else:
                scenelocs = gatherscenelocs()
                print(f"--------> sceneloc not found {scenelocname}")
                print(f"Possible scenelocs to use:")
                if len(scenelocs) == 0:
                    print(f"    origin")
                else:
                    for s in scenelocs:
                        print(f"    {s}")

                bpy.ops.wm.quit_blender()
                return None,None

        # test scenelocused
        self.addscenelocused(ia)


        # Add empty above camera 
        O.object.empty_add(type='PLAIN_AXES')
        empty = C.active_object
        empty.name = "shim_" + scenelocname + "_" + ia.exportbase.uuid
        empty.rotation_euler[2] = math.radians(scenelocrotation)
        empty.location = (0,0,0)
        empty.parent = cameraObject.parent
        cameraObject.parent = empty

        plane = None

        xfov = 90
        exportbase = ia.exportbase
        camera_frames = exportbase.camera.camera_frames
        if len(camera_frames) == 0:
            return None,None

        xfov = camera_frames[0].xFovDegrees
        ck = CompositeKey()
        ck.appendnode(context, ia, self.report)

        if ia.exportbase.cameraplane:
            ccip = CameraPlane.CreateCameraImagePlane()
            ccip.createImagePlaneForCamera(context, cameraObject, ia)
            plane = self._findplane(cameraObject)

        framecount = exportbase.timelinestartframe
        numframes = exportbase.endframe - exportbase.startframe + 1 # Inclusive
        endframecount = framecount + numframes

        # start_frame is 1 indexed. 
        startindex = 1
        for f in range(numframes):
            #print(f"framecount {framecount} frametime {frametime} fps: {fps}")
            C.scene.frame_set(framecount)
            self.setPrimCamera(camera_frames[f],exportbase.camera,cameraObject,plane,ia.depthperson,framecount)
            framecount = framecount + 1

        C.scene.frame_end = exportbase.timelinestartframe + numframes # Inclusive range
        C.scene.frame_set(exportbase.startframe)

        # Add Audio
        if exportbase.audiopath != "":
            audio_start_frame = exportbase.timelinestartframe-exportbase.startframe + 1
            shift_frames = int((exportbase.shift_amount * exportbase.fps) / 1000)
            audio_start_frame = audio_start_frame + shift_frames

            audio_end_frame = audio_start_frame + numframes

            # Ensure sequence editor exists
            if not C.scene.sequence_editor:
                C.scene.sequence_editor_create()

            # Blender 5.0+ uses strips instead of sequences
            seq_ed = C.scene.sequence_editor
            if hasattr(seq_ed, 'strips'):
                sequences = seq_ed.strips
            elif hasattr(seq_ed, 'sequences'):
                sequences = seq_ed.sequences
            else:
                raise AttributeError("Cannot find strips or sequences attribute in sequence_editor")

            sound_strip = sequences.new_sound(
                name="Sound",
                filepath=exportbase.audiopath,
                channel=1,
                frame_start=audio_start_frame,
            )

        self.import_scan(ia,cameraObject.parent)
        self.import_externalscan(ia,cameraObject.parent)

        #self.adjust_timeline_view(context)


        return cameraObject,plane


    def import_tracking_worker(self, context, ia:ImportArguments):
        C = context
        O = bpy.ops

        ia.pulluuid()
        if len(ia.exportbase.uuid) == 0:
            self.report({'INFO'}, "Missing uuid in filename")
            return {'FINISHED'}

        ia.shotname = ia.exportbase.uuid
        
        #if context.scene.AFTERBURNER.camerabackground_boolean: # Only used for testing
        #    ia.exportbase.timelinestartframe = 1
        #    ia.exportbase.startframe = 1
        #    ia.exportbase.endframe = 10000
        #    ia.exportbase.scanpath = ""

        if not self.import_file_tracking(context,ia):
            return {'CANCELLED'}


        #if context.scene.AFTERBURNER.camerabackground_boolean:
        #    self.set_camerabackground(context, ia,context.scene.camera)

        #w,h = ScreenResolution.GetResolution(True)
        w = ia.exportbase.renderwidth
        h = ia.exportbase.renderheight
        print(f"Scene rendering resolution set to {w}x{h}")
        C.scene.render.resolution_x = w
        C.scene.render.resolution_y = h

        return {'FINISHED'}
    
    def setup_color_management(self, ia: ImportArguments):
        """
        Sets the scene's View Transform based on the camera media properties.
        """
        scene = bpy.context.scene
        if not scene:
            return

        media_camera = ia.exportbase.media_camera
        target_transform = 'Standard' 

        # Case 1: The export explicitly specifies ACEScg.
        # NOTE: This requires the user to have the ACES OCIO config installed in Blender's preferences!
        if media_camera.colorspace == 'ACEScg':
            # The exact name can vary based on the OCIO config file.
            # 'ACES' is common, but check your UI for the exact string.
            # We will try to find a transform named 'ACES'.
            available_transforms = [item.identifier for item in scene.view_settings.bl_rna.properties['view_transform'].enum_items]
            if 'ACES' in available_transforms:
                target_transform = 'ACES'
                print("Found ACEScg media. Setting View Transform to ACES.")
            else:
                target_transform = 'Filmic'

        # Case 2: Standard LDR (Low Dynamic Range) footage like JPEG, PNG, or typical video files.
        else:
            target_transform = 'Standard'
        

        try:
            scene.view_settings.view_transform = target_transform
            print(f"Successfully set View Transform to: '{target_transform}'")
        except TypeError:
            print(f"Error: Could not set View Transform to '{target_transform}'. It may not be a valid option in the current OCIO config.")


    def execute(self, context):
        ImportOperator._instance = self

        print("Start of execute() import_tracking")
        ia = ImportArguments()
        myprops = context.scene.AFTERBURNER
        ia.exportbase = readserialized(myprops.exportfilepath)
        if not ia.exportbase:
            return {'CANCELLED'}
        
        self.setup_color_management(ia)

        myprops.renderdir = ia.exportbase.renderdirectory
        ia.depthperson = ia.exportbase.depth_amount == 0
        ia.sceneblendpath = myprops.sceneblendpath
        ia.scenetype = SceneType.get_type(myprops.scenetype)
        ia.runtype = RunScope.get_type(myprops.runtype)
        ia.pulluuid()
        d = Daily(ia)
        d.setup(True)

        #ia.depth_amount = myprops.depth_amount
        #ia.campath = myprops.campath
        #ia.scanpath = myprops.scanpath
        #ia.aipath = myprops.aipath
        #ia.aipatheroded = myprops.aipatheroded
        #ia.depthpath = myprops.depthpath
        #ia.audiopath = myprops.audiopath
        #ia.shift_amount = myprops.shift_amount
        #ia.fps = myprops.fps
        #ia.namesuffix = myprops.namesuffix
        #ia.colorspace = myprops.colorspace

        if ia.scenetype == SceneType.EMPTY:
            ia.composite = True
        
        r = self.import_tracking_worker(context, ia)
        myprops.cameraname = ia.cameraname

        code = r.pop()
        print(f"import_tracking_worker returned {code}")

        if code == 'FINISHED':
            d.finish()

        if hasattr(myprops, 'quickrender_boolean') and myprops.quickrender_boolean:
            print("Quick render requested after import.")
            d.fastrender()



        return r
