# ##### BEGIN GPL LICENSE BLOCK #####
#
#  This program is free software; you can redistribute it and/or
#  modify it under the terms of the GNU General Public License
#  as published by the Free Software Foundation; either version 2
#  of the License, or (at your option) any later version.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program; if not, write to the Free Software Foundation,
#  Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
#
# ##### END GPL LICENSE BLOCK #####


# https://blenderartists.org/t/scripts-create-camera-image-plane/580839


from asyncio import create_subprocess_exec
from genericpath import isfile
from tokenize import String
from xmlrpc.client import Boolean
import bpy
import os
from bpy_extras.image_utils import load_image
import math
import glob

from . import imageloading
from .argutil import ImportArguments
from . import version
from .exportbase import ExportCamera


    

def locatetemplatefile(templatefile:str):

    dirs = CompositeKey.kBlenderdirectory
    dirname = os.path.dirname(imageloading.__file__)
    dirs.append(dirname)

    for dir in dirs:
        files = glob.glob(dir + '/' + templatefile + "*.blend", recursive=False)
        if len(files) > 0:
            tpath = files[0]
            if os.path.isfile(tpath):
                return tpath

    print(f"Could not find {templatefile} in {dirs}")
    return None

# Manual compatibility mapping
COLORSPACE_MAP = {
    "ACEScg": [
        "ACEScg",
        "Utility - Linear - ACEScg",
        "Linear ACEScg"
    ],
    "sRGB": [
        "sRGB",
        "sRGB - Texture",
        "Output - sRGB"
    ],
    "Linear Rec.709": [
        "Utility - Linear - Rec.709",
        "Linear Rec709",
        "Linear Rec.709",
        "Rec709 (Linear)"
    ]
}

def get_available_colorspaces():
    """
    Retrieves the valid color space identifiers by accessing an existing image's
    colorspace_settings, or creating a temporary image if needed.
    """
    try:
        # Use an existing image if available.
        if bpy.data.images:
            img = bpy.data.images[0]
        else:
            # Create a temporary image (1x1 pixel).
            img = bpy.data.images.new("temp_colorspace", 1, 1)

        # Access the enum items for the 'name' property from this image's colorspace_settings.
        enum_items = img.colorspace_settings.bl_rna.properties['name'].enum_items
        available_colorspaces = [item.identifier for item in enum_items]

        # Remove the temporary image if we created one.
        if img.name == "temp_colorspace":
            bpy.data.images.remove(img)

        return available_colorspaces

    except Exception as e:
        print(f"Warning: Could not retrieve color space options: {e}")
        return []

def resolve_colorspace_name(target_name):
    """
    Maps a desired colorspace name (like 'ACEScg') to an actual valid Blender color space identifier.
    Returns the matched identifier or None if no match is found.
    """
    available_colorspaces = get_available_colorspaces()
    if not available_colorspaces:
        print("No available color spaces found.")
        return None

    aliases = COLORSPACE_MAP.get(target_name, [])
    for alias in aliases:
        for actual in available_colorspaces:
            if alias.lower() == actual.lower():
                return actual

    return None  # No match found

class CompositeKey:
    kGroupName = "Autoshot"
    kColorKeyerName = "ColorKeyer"
    kBlenderfilename=kGroupName
    kgrouppath=f"/NodeTree/{kGroupName}"
    kcolorkeyerpath=f"/NodeTree/{kColorKeyerName}"
    kBlenderdirectory=["d:/tmp/","c:/tmp/"]

    def __init__(self):
        self._islist = False
        self._isvalid = False
        self.batchrenderpath = ""
        self.outputpath = ""
        self.shootpath = ""

    @staticmethod
    def loadcameraimage(ia:ImportArguments,usepath:String,camera:Boolean,prefix:String=None,colorspace="sRGB"):
        path = None
        print(f"loadcameraimage {ia.shotname} {usepath} {camera}")

        if len(usepath) != 0:
            path = usepath
            isdir = not os.path.isfile(path)

        else:
            for trypath in ia.outputpathtexture:
                if camera:
                    (path,isdir) = imageloading.TextureFind(trypath,None,ia.shotname)
                else:
                    (path,isdir) = imageloading.DepthVideoFind(trypath,None,ia.shotname)
                if path != None:
                    break

        if path == None:
            print(f"No camera={camera} texture found in {ia.outputpathtexture} for {ia.shotname}")
            return None


        spec = None
        if isdir:
            if os.path.isdir(path):  # Could be not a file and not a dir if missing
                images = imageloading.load_imagesequence_fromdir(path, ia.exportbase.startframe, ia.exportbase.sequencestartframe,ia.exportbase.timelinestartframe,prefix)
                for img_spec in images:
                    spec = img_spec
                    #self.apply_texture_options(texture,ia.timeline_start_frame, img_spec)
                    break  # Take only first of list
        else:
            images = imageloading.load_imagesexact(path,"", ia.exportbase.startframe, ia.exportbase.timelinestartframe)

            for img_spec in images:
                spec = img_spec
                break  # Take only first of list

        if spec:
            print(f"colorspace_settings.name: {spec.image.colorspace_settings.name}  requested colorspace: {colorspace}")
            
            resolved = resolve_colorspace_name(colorspace)
            if resolved:
                try:
                    spec.image.colorspace_settings.name = resolved
                    print(f"Set colorspace to '{resolved}'")
                except Exception as e:
                    print(f"Failed to set colorspace to '{resolved}': {e}")
            else:
                print(f"Could not resolve a matching colorspace for '{colorspace}'")


        return spec

    @staticmethod
    def setResolutionFromImage(image):
        w,h = image.size
        extension = os.path.splitext(image.filepath)[1]
        isPng = extension == ".png" or extension == ".PNG"
        print(f"extension {extension} isPng {isPng}")
        
        if w == 0:
            print(f"setResolutionFromImage {image.filepath} missing size")
            # image.reload() Why doesn't this work?
            img = bpy.data.images.load(image.filepath)
            w,h = img.size
            print(f"after bpy.data.images.load {image.filepath} {w}x{h}")

        if w != 0:
            print(f"setResolutionFromImage {image.filepath} {w}x{h}")
            ScreenResolution.SetResolution(w,h)



    def safelink(self,links,outputnode,outputname,inputnode,inputname):
        if outputname in outputnode.outputs:
            output_socket = outputnode.outputs[outputname]
            if inputname in inputnode.inputs:
                input_socket = inputnode.inputs[inputname]
                links.new(output_socket, input_socket)
            else:
                print(f"inputname {inputname} missing")
                print(f"inputs {inputnode.inputs.keys()}")
        else:
            print(f"outputname {outputname} missing")
            print(f"outputs {outputnode.outputs.keys()}")


    def verifyblenderrealtimecompositor(self, reportcallback, ia:ImportArguments):
        pref = bpy.context.preferences
        dosave = False

        print(f"show_developer_ui is {pref.view.show_developer_ui}")
        if not pref.view.show_developer_ui:
            print("Set show_developer_ui to True")
            pref.view.show_developer_ui = True
            dosave = True

        #pref = bpy.context.preferences.experimental
        #if hasattr(pref,"use_realtime_compositor"):
        #    reportcallback({'INFO'}, 'Blender version supports realtime compositor')
        #    print(f"PreferencesExperimental.use_realtime_compositor is {pref.use_realtime_compositor}")
        #    if not pref.use_realtime_compositor:
        #        print("Set use_realtime_compositor to True")
        #        pref.use_realtime_compositor = True
        #        dosave = True
        #else:
        #    reportcallback({'INFO'}, 'Blender version does not support realtime compositor')

        if  dosave:
            print("Saving user preferences")
            bpy.ops.wm.save_userpref()

        # Preview the compositor output inside the viewport
        for area in bpy.context.screen.areas: 
            if area.type == 'VIEW_3D':
                for space in area.spaces: 
                    if space.type == 'VIEW_3D':
                        print(f"Setting 'MATERIAL' instead of {space.shading.type} ")
                        space.shading.type = 'MATERIAL'
                        if ia.composite and hasattr(space.shading,"use_compositor"):
                            print(f"use_compositor is {space.shading.use_compositor} {type(space.shading.use_compositor)} ")

                            if type(space.shading.use_compositor) is bool:
                                print(f"Setting use_compositor true instead of {space.shading.use_compositor} ")
                                space.shading.use_compositor = True
                            else:
                                print(f"Setting use_compositor CAMERA instead of {space.shading.use_compositor} ")
                                space.shading.use_compositor = 'CAMERA'
                                #if ia.aipath == "":  # For cine case
                                #    space.shading.use_compositor = 'DISABLED'

    def appendnode(self, context, ia:ImportArguments,reportcallback):
        #start_time = datetime.datetime.now()
        #cont = context.area.type
        print("appendnode start")

        self.verifyblenderrealtimecompositor(reportcallback,ia)

        livekey = bpy.context.scene.AFTERBURNER.livekey

        fullpath = locatetemplatefile(CompositeKey.kBlenderfilename)
        if fullpath == None:
            return

        print(f"Template {fullpath}")
        fdir = os.path.dirname(fullpath)
        filename = os.path.basename(fullpath)

        if ia.composite:
            # Compositing Node
            appendpath = filename + CompositeKey.kgrouppath
            print(f"Appending {fdir} {appendpath}")
            bpy.ops.wm.append(
                filename=appendpath, directory=fdir
            )

        # ColorKeyer Node
        appendpath = filename + CompositeKey.kcolorkeyerpath
        print(f"Appending {fdir} {appendpath}")
        bpy.ops.wm.append(
            filename=appendpath, directory=fdir
        )


        # XXX We may need to check and delete existing nodetrees named CompositeKey
        # # Set the name of the node group to delete
        # node_group_name = "My Node Group"
        # Get a reference to the node group
        # node_group = bpy.data.node_groups[node_group_name]

        # Remove the node group from all nodes that are using it
        #for node in node_group.users:
        #    node.node_group = None

        # Remove the node group from Blender's data
        #bpy.data.node_groups.remove(node_group)
 
        groupname = CompositeKey.kGroupName
        if not groupname in bpy.data.node_groups:
            print(f"{groupname} not found")
            return

        context.scene.use_nodes = True
        node_tree = context.scene.node_tree
        top_group = node_tree.nodes.new("CompositorNodeGroup")
        top_group.node_tree = bpy.data.node_groups[groupname]
        top_group.name = "CompositeKey"
        top_group.location = -200,300
        print(f"top_group {top_group}")
        cameraimage = top_group.node_tree.nodes['CameraImage']
        print(f"cameraimage {cameraimage}")
        print(f"image {cameraimage.image}")

        matteimage = top_group.node_tree.nodes['MatteImage']

            #self.ia.aipath = self.stubargs["aipath"]
            #self.ia.depthpath = self.stubargs["depthpath"]

        spec = CompositeKey.loadcameraimage(ia,ia.exportbase.media_camera.path,True)    
        print(f"Using spec {spec}")

        if spec != None:
            cameraimage.image = spec.image
            CompositeKey.setResolutionFromImage(cameraimage.image)
            cameraimage.frame_start = ia.exportbase.timelinestartframe
            cameraimage.frame_offset = spec.frame_offset + (ia.exportbase.startframe - 1)
            cameraimage.frame_duration = spec.frame_duration
            # Math is wrong if we want this check
            #if ia.end_frame > cameraimage.frame_duration:
            #    ia.end_frame = cameraimage.frame_duration

        if ia.exportbase.media_aimatte.path != "":
            spec = CompositeKey.loadcameraimage(ia,ia.exportbase.media_aimatte.path,True)    
            print(f"Using spec {spec}")

            if spec != None:
                matteimage.image = spec.image
                matteimage.image.alpha_mode = 'NONE'
                matteimage.frame_start = ia.exportbase.timelinestartframe
                matteimage.frame_offset = spec.frame_offset +  (ia.exportbase.startframe - 1)
                matteimage.frame_duration = spec.frame_duration


        depthimage = top_group.node_tree.nodes['DepthImage']
        spec = CompositeKey.loadcameraimage(ia,ia.exportbase.media_depth.path,False)    
        print(f"Using spec {spec}")

        if spec != None:
            depthimage.image = spec.image
            depthimage.image.colorspace_settings.name = 'Non-Color'
            depthimage.frame_start = ia.exportbase.timelinestartframe
            depthimage.frame_offset = spec.frame_offset +  (ia.exportbase.startframe - 1)
            depthimage.frame_duration = spec.frame_duration

        #if not livekey:
        #    cgrenderimage = node_tree.nodes['cgrender']
        #    spec = CompositeKey.loadcameraimage(ia,ia.cgrenderpath,False)    
        #    print(f"Using spec {spec}")
#
        #    if spec != None:
        #        cgrenderimage.image = spec.image
        #        cgrenderimage.frame_start = ia.exportbase.timelinestartframe
        #        cgrenderimage.frame_offset = ia.exportbase.timelinestartframe - 1
         #       cgrenderimage.frame_duration = spec.frame_duration


        # Set switch nodes for type of composite
        nodes = bpy.data.node_groups[groupname].nodes
        if "Real Time Depth" in nodes:
            nodes["Real Time Depth"].check = False
        if "Real Time FG" in nodes:
            nodes["Real Time FG"].check = livekey  # Should always be true now


        # Set the scale of the transform node to match the simulated focallength
        transform_node_name = "Transform.001"

        transform_node = bpy.data.node_groups[groupname].nodes.get(transform_node_name)
        if transform_node != None:
            transform_node.inputs["Scale"].default_value = ia.exportbase.simfocalscale

        links = node_tree.links


        render_node = node_tree.nodes['Render Layers']
        render_node.location =  -500, 300
        # Get rid of the default link from render layer to output
        l = render_node.outputs[0].links[0]
        links.remove(l)
        

        # Save and set to CYCLES to get all our render layers
        current_renderengine = context.scene.render.engine     
        context.scene.render.engine = 'CYCLES'

        # Turn on needed rendering
        for vl in context.scene.view_layers:
            vl.use_pass_z = True
            vl.use_pass_mist = True
            vl.use_pass_cryptomatte_object = True
            vl.cycles.use_pass_shadow_catcher = True

        context.scene.render.engine = current_renderengine

        # Link the renderimage into the input of topgroup
        if livekey:
            for rname,gname in {'Alpha':'Alpha','Depth':'Depth','Mist':'Mist', 'Image':'RenderImage', 
                        'CryptoObject00':'Crypto 00','CryptoObject01':'Crypto 01','CryptoObject02':'Crypto 02'}.items():
                self.safelink(links,render_node,rname,top_group,gname)
        else:
            for rname,gname in {'Alpha':'Alpha','Depth':'Depth','Mist':'Mist', 'Combined':'RenderImage'}.items():
                self.safelink(links,render_node,rname,top_group,gname)


        # Link the output of topgroup into the Composite
        composite_node = node_tree.nodes['Composite']
        composite_node.location = 0,300
        output_socket = composite_node.inputs['Image']
        input_socket = top_group.outputs['OutputImage']
        links.new(output_socket, input_socket)

        return 


class ScreenResolution:
    ScreenWidth=0
    ScreenHeight=0

    # Called by operator or from tracking
    @staticmethod
    def SetResolution(w:int,h:int):
        assert(w != 0)
        # Don't let a slightly off image height change scene resolution
        if h >= 1080 and h <= 1100:
            w = 1920
            h = 1080

        if ScreenResolution.ScreenWidth == 0:
            ScreenResolution.ScreenWidth = w
            ScreenResolution.ScreenHeight = h
            print(f"Changing scene resolution to {w}x{h}")

    @staticmethod
    def GetResolution(takedefault:bool = False):
        if ScreenResolution.ScreenWidth == 0 and takedefault:
            ScreenResolution.ScreenWidth = 1920
            ScreenResolution.ScreenHeight = 1080
            print(f"Using default resolution {ScreenResolution.ScreenWidth}x{ScreenResolution.ScreenHeight}")
        
        return ScreenResolution.ScreenWidth,ScreenResolution.ScreenHeight

