from enum import Enum
import json
import sys

class Keyer:
    """A class to hold and manage keyer settings."""
    def __init__(self, settings_dict=None):
        if settings_dict is None:
            settings_dict = {}
        # Set defaults first
        self.KeyColorRed = 0.0
        self.KeyColorGreen = 1.0
        self.KeyColorBlue = 0.0
        self.ClipBlack = 0.0
        self.ClipWhite = 1.0
        self.WeightRed = 0.333
        self.WeightOther = 0.333
        self.RemoveSpill = True
        self.GreenBlue = True
        self.AlphaOffset = 0.0
        self.BlurSigma = 0.0
        self.AlphaThreshold = 0.5
        # Overwrite with any provided settings
        self.from_dict(settings_dict)

    def from_dict(self, data: dict):
        """Populates attributes from a dictionary."""
        for key, value in data.items():
            if hasattr(self, key):
                setattr(self, key, value)

    def to_dict(self) -> dict:
        """Converts attributes to a dictionary."""
        return self.__dict__

    def get_color_as_hex(self) -> str:
        """Returns the key color as a hex string (e.g., #RRGGBB)."""
        r = int(self.KeyColorRed * 255)
        g = int(self.KeyColorGreen * 255)
        b = int(self.KeyColorBlue * 255)
        return f"#{r:02x}{g:02x}{b:02x}"

    def set_color_from_hex(self, hex_color: str):
        """Sets the key color from a hex string."""
        hex_color = hex_color.lstrip('#')
        rgb = tuple(int(hex_color[i:i+2], 16) for i in (0, 2, 4))
        self.KeyColorRed = rgb[0] / 255.0
        self.KeyColorGreen = rgb[1] / 255.0
        self.KeyColorBlue = rgb[2] / 255.0


class MediaType(Enum):
    ORIGINAL = 'Original File'
    PNG = 'Extracted PNGs'
    JPEG = 'Extracted JPEGs'
    EXR = 'Extracted EXRs'

    @staticmethod
    def get_type(media_type_str):
        for media_type in MediaType:
            if media_type.value == media_type_str:
                return media_type
        return MediaType.ORIGINAL  # return ORIGINAL if no matching member is found


class MediaEntry:
    def __init__(
        self,
        path: str = None,  # Relative or Absolute path to either sequence dir or video file. If "" media not used.
        template : str = None,  # camo_{0:05d}.png
        mediatype : MediaType = MediaType.PNG,
        colorspace : str = None
    ):
        self.path = path
        self.template = template
        self.mediatype = mediatype
        self.colorspace = colorspace


class CameraFrame:
    def __init__(
        self,
        xFovDegrees:float = 0,
        cx:float = 0,
        cy:float = 0,
        depth:float = 0,
        pose:[float] = [],  # Blender coordinate system
        rotmat:[float] = []
    ):
        self.xFovDegrees = xFovDegrees
        self.cx = cx
        self.cy = cy
        self.depth = depth  # Depth of person in meters if detected. 0 if no person. Use last good value starting with default of 2m.
        # Blender coordinates. Right handed Z up.
        self.pose = pose
        self.rotmat = rotmat

class SceneLoc:
    def __init__(
        self,
        name:str = "",
        rotation:int = 0,
        pose:[float] = [0,0,0],  # Blender coordinate system
        rotmat:[] = [[1,0,0],[0,1,0],[0,0,1]]
    ):
        self.name = name
        self.rotation = rotation  # Extra rotation angle applied by GUI
        self.pose = pose
        self.rotmat = rotmat

class ExportCamera:
    def __init__(
        self,
        width:int,
        height:int,
        sensorwidth:float,
        focallength:float,
        undistort_stmap:str,
        camera_frames:[CameraFrame]
    ):
        self.width = width
        self.height = height
        self.sensorwidth = sensorwidth
        self.focallength = focallength  # Focal length in mm
        self.undistort_stmap = undistort_stmap # Path to exr with stmap
        self.xfovdegrees = 0.0
        self.camera_frames = camera_frames
        self.k1 = 0.0
        self.k2 = 0.0
        self.k3 = 0.0
        self.p1 = 0.0
        self.p2 = 0.0
        self.fx = 0.0
        self.fy = 0.0
        self.cx = 0.0
        self.cy = 0.0
        self.model = "SIMPLE_RADIAL"



class ExportBase:
    def __init__(
        self,
        sceneloc:str,   # Deprecated.  in scenelocused
        scenelocrotation:int, # Deprecated. in scenelocused
        timelinestartframe:int, # 1 based index of destination frame in target system
        startframe:int, # 1 based index of starting frame in media
        endframe:int,  # 1 based index of end frame in media.
        sequencestartframe:int, # Frameoffset for image sequence
        renderwidth:int,
        renderheight:int,
        depth_amount:int,  # If non-zero. Depth in meters of estimate of subject in camera
        uuid:str,
        jobname:str,
        shortname:str,
        fps:float,
        namesuffix:str,  # "" or "_cine" cosmetic
        renderdirectory:str,
        cameraplane:bool,  # True if we want to add a plane holding the camera media in the target system
        scanpath:str,  # Abosolute path to the USD file of a room scan
        externalscanpath:str,  # Abosolue path to the OBJ file of a room scan. Scale 1m
        audiopath:str,  # Full audio of track stripped from the camera video
        shift_amount:int, # Tracking shift to calculate matching audio time shift
        simfocalscale:float, # Float Given the simulated focal length in the tracking compute the scale of the camera footage needed to create the focal length at our target sensorwidth
        scenelocused:SceneLoc,
        camera:ExportCamera,
        media_camera:MediaEntry,
        media_depth:MediaEntry,
        media_aimatte: MediaEntry,
        keyer: Keyer = None  # Add the new keyer parameter
    ):
        self.sceneloc = sceneloc
        self.scenelocrotation = scenelocrotation
        self.timelinestartframe = timelinestartframe
        self.startframe = startframe
        self.endframe = endframe
        self.sequencestartframe = sequencestartframe
        self.renderwidth = renderwidth
        self.renderheight = renderheight
        self.depth_amount = depth_amount
        self.uuid = uuid
        self.jobname = jobname
        self.shortname = shortname
        self.fps = fps
        self.namesuffix = namesuffix
        self.renderdirectory = renderdirectory
        self.cameraplane = cameraplane
        self.scanpath = scanpath
        self.externalscanpath = externalscanpath
        self.audiopath = audiopath
        self.shift_amount = shift_amount
        self.simfocalscale = simfocalscale
        self.scenelocused = scenelocused
        self.camera = camera
        self.media_camera = media_camera
        self.media_depth = media_depth
        self.media_aimatte = media_aimatte
        self.keyer = keyer if keyer is not None else Keyer()


def serialize(obj):
    """
    Serialize a Python object into a JSON-serializable format, including class type information.
    """
    if obj is None:
        return None
    elif isinstance(obj, Enum):
        return {"__enum__": str(obj)}
    elif isinstance(obj, (int, float, str, bool)):
        return obj
    elif isinstance(obj, dict):
        return {k: serialize(v) for k, v in obj.items()}
    elif isinstance(obj, list):
        return [serialize(i) for i in obj]
    elif hasattr(obj, '__dict__'):
        data = {'__class__': type(obj).__name__, '__attributes__': serialize(obj.__dict__)}
        return data
    else:
        raise TypeError(f"Type {type(obj)} not serializable")

# Add all the classes you need to handle here
deserializeclass_map = {'ExportBase': ExportBase,
            'MediaEntry': MediaEntry,
             'ExportCamera': ExportCamera,
             'CameraFrame': CameraFrame,
             'SceneLoc': SceneLoc,
             'Keyer': Keyer,
            }

deserializeenum_registry = {
    'MediaType': MediaType,
    }

def deserialize(data):
    """
    Deserialize JSON data into a Python object using class type information.
    """
    if data is None:
        return None
    elif isinstance(data, dict) and "__enum__" in data:
        enum_name, member_name = data["__enum__"].split(".")
        return deserializeenum_registry[enum_name][member_name]
    elif isinstance(data, dict) and '__class__' in data:
        class_name = data['__class__']
        if class_name in deserializeclass_map:
            cls = deserializeclass_map[class_name]
            obj = cls.__new__(cls)
            obj.__dict__ = deserialize(data['__attributes__'])
            return obj
        else:
            raise ValueError(f"Unknown class {class_name}")
    elif isinstance(data, dict):
        return {k: deserialize(v) for k, v in data.items()}
    elif isinstance(data, list):
        return [deserialize(i) for i in data]
    else:
        return data

def writeserialized(path, obj):
    try:
        file = open(path, 'w')
        serialized = serialize(obj)
        json.dump(serialized, file, indent=4)
        file.close()
    except IOError:
        print(f"Could not open {path}")

def readserialized(path):
    try:
        file = open(path, 'r')
        data_loaded = json.load(file)
        deserialized_obj = deserialize(data_loaded)
        file.close()
        return deserialized_obj
    except IOError:
        print(f"Could not open {path}")
        return None

# Read an exported json track and dump just the first 2 frames
def main(filename):
    export = readserialized(filename)
    if export is not None:
        export.camera.camera_frames = export.camera.camera_frames[0:1]
        serialized = serialize(export)
        print(json.dumps(serialized, indent=4))
    else:
        print(f"No object found at {filename}")

if __name__ == "__main__":
    #main("D:/testsets/askill_SC101_TK003_A001_1080P-709__46CA1E60D9/Project/Sequences/myproj_2023-06-14/myproj_SC101_TK003_A001_1080P-709__46CA1E60D9/tmp/export.json")
    #sys.exit(0)

    if len(sys.argv) != 2:
        print("Usage: exportbase.py <filename>")
        sys.exit(1)

    filename = sys.argv[1]
    main(filename)
