import bpy
import math
import mathutils
import csv
import os
import bmesh
import shutil
import subprocess


class GenerateCameraPathOperator(bpy.types.Operator):
    """Generate Multiple Camera Circles with Corrected 'XYZ' Euler Rotations"""
    bl_idname = "object.generate_camera_path"
    bl_label = "Generate Camera Path"
    bl_options = {'REGISTER', 'UNDO'}
    
    def execute(self, context):
        # Parameters
        N = 60  # Number of camera positions along each circle
        radius = 1.0  # Radius of the circles
        focal_length = 30.0  # Camera focal length in mm

        # Define the circle parameters (height and pitch angle)
        # Each entry is (height, pitch angle in degrees)
        circles = [
            (1.0, -10.0),   # Circle at 1 meter height, looking up 10 degrees
            (2.0, 0.0),    # Circle at 2 meters height, looking straight ahead
            (3.0, -20.0)   # Circle at 3 meters height, looking down 20 degrees
        ]
        
        scene = context.scene
        obj = context.active_object
        if obj is None:
            self.report({'ERROR'}, "No active object selected.")
            return {'CANCELLED'}
        
        obj_location = obj.location.copy()
        
        # Create a new camera
        cam_data = bpy.data.cameras.new("SplatCamera")
        cam_data.lens = focal_length
        cam_obj = bpy.data.objects.new("SplatCamera", cam_data)
        scene.collection.objects.link(cam_obj)
        scene.camera = cam_obj  # Set the new camera as the active camera

        frame_number = 1  # Starting frame number

        for height, pitch_adjust in circles:
            # Calculate positions along the circle
            for i in range(N):
                angle = (2 * math.pi / N) * i
                x = obj_location.x + radius * math.cos(angle)
                y = obj_location.y + radius * math.sin(angle)
                #z = obj_location.z + height
                z = height

                # Set the frame
                scene.frame_set(frame_number)
                frame_number += 1  # Increment frame number
                print(f"frame {frame_number}")
                # Set camera location
                cam_location = mathutils.Vector((x, y, z))
                cam_obj.location = cam_location
                cam_obj.keyframe_insert(data_path="location", index=-1)

                # Compute rotation to look at the object center
                # Calculate the direction vector pointing to the center
                direction = (obj_location - cam_location).normalized()

                # Create a rotation quaternion that aligns -Z with the direction vector
                rotation_quat = direction.to_track_quat('-Z', 'Y')

                # Convert the quaternion to Euler angles in 'XYZ' order
                rotation_euler = rotation_quat.to_euler('XYZ')

                # Adjust the pitch angle (rotation around X-axis)
                rotation_euler.x = math.radians(90 + pitch_adjust)

                # Set camera rotation
                cam_obj.rotation_mode = 'XYZ'
                cam_obj.rotation_euler = rotation_euler
                cam_obj.keyframe_insert(data_path="rotation_euler", index=-1)

        self.report({'INFO'}, "Camera paths generated successfully.")
        return {'FINISHED'}


class ExportCameraPosesOperator(bpy.types.Operator):
    """Export Camera Poses and Intrinsics to CSV with Rendered Filenames"""
    bl_idname = "export_camera.poses_to_csv_with_render_filepath"
    bl_label = "Export Camera Poses to CSV (Using Render Filepath)"
    bl_options = {'REGISTER'}

    filepath: bpy.props.StringProperty(subtype="FILE_PATH")

    def execute(self, context):
        # Get the selected camera
        cam = context.active_object
        if cam is None or cam.type != 'CAMERA':
            self.report({'ERROR'}, "Please select a camera object.")
            return {'CANCELLED'}

        scene = context.scene
        frame_start = scene.frame_start
        frame_end = scene.frame_end

        # Extract camera intrinsic parameters
        focal_length_px, principal_x, principal_y = self.get_camera_intrinsics(cam, scene)

        # Distortion coefficients (assuming zero unless specified)
        K1 = K2 = K3 = K4 = T1 = T2 = 0.0

        # Get the render filepath components
        base_name, ext = self.get_render_filepath_components(scene)

        # Open the CSV file for writing
        with open(self.filepath, 'w', newline='') as csvfile:
            csvwriter = csv.writer(csvfile)

            # Write the header
            csvwriter.writerow([
                "#name", "x", "y", "alt",
                "heading", "pitch", "roll",
                "f", "px", "py",
                "k1", "k2", "k3", "k4", "t1", "t2"
            ])

            # Iterate over the frames
            for frame in range(frame_start, frame_end + 1):
                scene.frame_set(frame)

                # Construct the image filename based on the render filepath
                image_name = self.get_image_name(base_name, ext, frame)

                # Get camera extrinsic parameters (moved inside the loop)
                # Get camera world location
                cam_matrix_world = cam.matrix_world.copy()
                cam_location = cam_matrix_world.to_translation()

                x = cam_location.x
                y = cam_location.y
                alt = cam_location.z  # Assuming Z is altitude

                # Get camera rotation matrix
                cam_rotation_world = cam_matrix_world.to_3x3()

                # Convert rotation matrix to Euler angles in 'YXZ' order (Heading, Pitch, Roll)
                euler = cam_rotation_world.to_euler('YXZ')

                # Convert Euler angles from radians to degrees
                heading = math.degrees(euler.z)
                pitch = math.degrees(euler.y)
                roll = math.degrees(euler.x)

                # Adjust heading, pitch, roll as needed
                heading = heading % 360
                pitch = pitch % 360
                roll = roll % 360

                # Write the row to CSV
                csvwriter.writerow([
                    image_name,
                    x, y, alt,
                    heading, pitch, roll,
                    focal_length_px,
                    principal_x, principal_y,
                    K1, K2, K3, K4, T1, T2
                ])

        self.report({'INFO'}, f"Camera poses exported to {self.filepath}")
        return {'FINISHED'}

    def get_camera_intrinsics(self, cam, scene):
        # Camera intrinsic parameters
        cam_data = cam.data
        focal_length_mm = cam_data.lens  # Focal length in mm
        sensor_width_mm = cam_data.sensor_width  # Sensor width in mm
        sensor_height_mm = cam_data.sensor_height  # Sensor height in mm

        # Assuming images are rendered at the scene's resolution
        resolution_x_px = scene.render.resolution_x
        resolution_y_px = scene.render.resolution_y
        pixel_aspect = scene.render.pixel_aspect_x / scene.render.pixel_aspect_y

        # Adjust sensor size for pixel aspect ratio
        sensor_width_mm *= pixel_aspect

        # Principal point (assuming center of the image)
        principal_x = resolution_x_px / 2
        principal_y = resolution_y_px / 2

        # Focal length in pixels
        focal_length_px = (focal_length_mm / sensor_width_mm) * resolution_x_px

        return focal_length_px, principal_x, principal_y

    def get_render_filepath_components(self, scene):
        # Get the base filepath and file extension from the render settings
        render_filepath = bpy.path.abspath(scene.render.filepath)
        base_dir = os.path.dirname(render_filepath)
        base_name = os.path.basename(render_filepath)
        base_name, ext = os.path.splitext(base_name)

        # Handle frame placeholders in the base name
        base_name = self.process_base_name(base_name)

        return base_name, ext

    def process_base_name(self, base_name):
        # If the base name contains a frame placeholder, we need to handle it
        if "%d" in base_name or "%0" in base_name:
            # Do nothing; the frame number will be inserted automatically
            pass
        elif "#" in base_name:
            # Do nothing; the frame number will be inserted automatically
            pass
        else:
            # If no frame number placeholder, append one
            base_name += "%04d"

        return base_name

    def get_image_name(self, base_name, ext, frame_number):
        image_name = base_name
        # Replace frame placeholders with actual frame numbers
        if "%" in image_name:
            image_name = image_name % frame_number
        elif "#" in image_name:
            num_hashes = image_name.count("#")
            image_name = image_name.replace("#" * num_hashes, f"{frame_number:0{num_hashes}d}")
        else:
            # Append frame number if no placeholder is present
            image_name += f"{frame_number:04d}"

        # Add the file extension
        image_name += ext

        return image_name

    def invoke(self, context, event):
        # Use the file browser to select the output CSV file
        context.window_manager.fileselect_add(self)
        return {'RUNNING_MODAL'}


class ExportCameraPosesToColmap(bpy.types.Operator):
    """Export cameras.txt and images.txt for COLMAP to render directory"""
    bl_idname = "export.colmap_camera_poses"
    bl_label = "Export Camera Poses to COLMAP"


    def write_colmap_cameras(self,output_dir, camera_data):
        """Writes the COLMAP cameras.txt file."""
        cameras_path = os.path.join(output_dir, "cameras.txt")
        with open(cameras_path, "w") as f:
            f.write("# Camera list with one line of data per camera:\n")
            f.write("#   CAMERA_ID, MODEL, WIDTH, HEIGHT, PARAMS[]\n")
            f.write("# Number of cameras: 1\n")
            
            for camera_id, (width, height, focal_length) in enumerate(camera_data):
                f.write(f"{camera_id + 1} SIMPLE_PINHOLE {width} {height} {focal_length} {width / 2} {height / 2}\n")

    def write_colmap_images(self,output_dir, image_data):
        """Writes the COLMAP images.txt file."""
        images_path = os.path.join(output_dir, "images.txt")
        with open(images_path, "w") as f:
            f.write("# Image list with two lines of data per image:\n")
            f.write("#   IMAGE_ID, QW, QX, QY, QZ, TX, TY, TZ, CAMERA_ID, NAME\n")
            f.write("#   POINTS2D[] as (X, Y, POINT3D_ID)\n")
            f.write(f"# Number of images: {len(image_data)}\n")

            for image_id, (name, quaternion, translation, camera_id) in enumerate(image_data):
                qw, qx, qy, qz = quaternion.w, quaternion.x, quaternion.y, quaternion.z
                tx, ty, tz = translation.x, translation.y, translation.z
                f.write(f"{image_id + 1} {qw} {qx} {qy} {qz} {tx} {ty} {tz} {camera_id + 1} {name}\n")
                f.write("\n")  # Write an extra blank line for missing POINTS2D

    def get_render_filepath_components(self, scene):
        # Get the base filepath from the render settings
        render_filepath = bpy.path.abspath(scene.render.filepath)
        base_dir = os.path.dirname(render_filepath)
        base_name = os.path.basename(render_filepath)
        base_name, ext = os.path.splitext(base_name)

        # If no extension is provided, derive it from the render file format
        if not ext:
            ext = self.get_extension_from_format(scene.render.image_settings.file_format)

        # Handle frame placeholders in the base name
        base_name = self.process_base_name(base_name)

        return base_name, ext, base_dir

    def get_extension_from_format(self, file_format):
        # Map Blender's file formats to common extensions
        format_to_ext = {
            'BMP': '.bmp',
            'IRIS': '.rgb',
            'PNG': '.png',
            'JPEG': '.jpg',
            'JPEG2000': '.jp2',
            'TARGA': '.tga',
            'TARGA_RAW': '.tga',
            'CINEON': '.cin',
            'DPX': '.dpx',
            'OPEN_EXR_MULTILAYER': '.exr',
            'OPEN_EXR': '.exr',
            'HDR': '.hdr',
            'TIFF': '.tif',
            'AVI_JPEG': '.avi',
            'AVI_RAW': '.avi',
            'FRAMESERVER': '.fs',
            'FFMPEG': '.mp4',  # Defaulting to .mp4 for FFMPEG
        }
        return format_to_ext.get(file_format, '')

    def process_base_name(self, base_name):
        # If the base name contains a frame placeholder, we need to handle it
        if "%d" in base_name or "%0" in base_name:
            # Do nothing; the frame number will be inserted automatically
            pass
        elif "#" in base_name:
            # Do nothing; the frame number will be inserted automatically
            pass
        else:
            # If no frame number placeholder, append one
            base_name += "%04d"

        return base_name

    def get_camera_intrinsics(self, cam, scene):
        # Camera intrinsic parameters
        cam_data = cam.data
        focal_length_mm = cam_data.lens  # Focal length in mm
        sensor_width_mm = cam_data.sensor_width  # Sensor width in mm
        sensor_height_mm = cam_data.sensor_height  # Sensor height in mm

        # Assuming images are rendered at the scene's resolution
        resolution_x_px = scene.render.resolution_x
        resolution_y_px = scene.render.resolution_y
        pixel_aspect = scene.render.pixel_aspect_x / scene.render.pixel_aspect_y

        # Adjust sensor size for pixel aspect ratio
        sensor_width_mm *= pixel_aspect

        # Principal point (assuming center of the image)
        principal_x = resolution_x_px / 2
        principal_y = resolution_y_px / 2

        # Focal length in pixels
        focal_length_px = (focal_length_mm / sensor_width_mm) * resolution_x_px

        return (focal_length_px, principal_x, principal_y,
                focal_length_mm, sensor_width_mm, sensor_height_mm)

    def export_colmap_files(self,context):
        """Main function to export camera and image files."""
        camera_data = []
        image_data = []

        cam = context.active_object
        if cam is None or cam.type != 'CAMERA':
            raise ValueError("Please select a camera object.")

        scene = context.scene
        render = scene.render
        resolution_x = render.resolution_x
        resolution_y = render.resolution_y

        intrinsic_params = self.get_camera_intrinsics(cam, scene)
        (focal_length_px, principal_x, principal_y,
         focal_length_mm, sensor_width_mm, sensor_height_mm) = intrinsic_params

        camera_data.append((resolution_x, resolution_y, focal_length_px))

        base_name, ext, base_dir = self.get_render_filepath_components(scene)
        output_dir = base_dir

        # Ensure the output directory exists
        if not os.path.exists(output_dir):
            os.makedirs(output_dir)

        frame_start = scene.frame_start
        frame_end = scene.frame_end

        # Loop over the frames
        for frame in range(frame_start, frame_end + 1):
            scene.frame_set(frame)

            # Construct the image filename based on the render filepath
            image_name = self.get_image_name(base_name, ext, frame)

            # Get world transform
            source_transform = cam.matrix_local.copy()

            # Adjust Blender transform to COLMAP format
            translation, rotation = adjust_to_colmap_format(source_transform)

            # Adjust for COLMAP's coordinate system
            image_data.append((image_name, rotation, translation, 0))

        self.write_colmap_cameras(output_dir, camera_data)
        self.write_colmap_images(output_dir, image_data)

    def get_image_name(self, base_name, ext, frame_number):
        image_name = base_name
        # Replace frame placeholders with actual frame numbers
        if "%" in image_name:
            image_name = image_name % frame_number
        elif "#" in image_name:
            num_hashes = image_name.count("#")
            image_name = image_name.replace("#" * num_hashes, f"{frame_number:0{num_hashes}d}")
        else:
            # Append frame number if no placeholder is present
            image_name += f"{frame_number:04d}"

        # Add the file extension
        image_name += ext

        return image_name

    def execute(self, context):
        # Get the selected camera
        cam = context.active_object
        if cam is None or cam.type != 'CAMERA':
            self.report({'ERROR'}, "Please select a camera object.")
            return {'CANCELLED'}

        try:
            self.export_colmap_files(context)
            self.report({'INFO'}, f"Exported COLMAP files")
            return {'FINISHED'}
        except ValueError as e:
            self.report({'ERROR'}, str(e))
            return {'CANCELLED'}

class ExportCameraPosesToXMP(bpy.types.Operator):
    """Export Camera Poses and Intrinsics to XMP files for RealityCapture"""
    bl_idname = "export_camera.poses_to_xmp"
    bl_label = "Export Camera Poses to XMP (Using Render Filepath)"
    bl_options = {'REGISTER', 'UNDO'}

    #output_directory: bpy.props.StringProperty(
    #    name="Output Directory",
    #    description="Directory to save XMP files",
    #    subtype='DIR_PATH',
    #    default=""
    #)
    output_directory = ""

    def execute(self, context):
        # Get the selected camera
        cam = context.active_object
        if cam is None or cam.type != 'CAMERA':
            self.report({'ERROR'}, "Please select a camera object.")
            return {'CANCELLED'}

        scene = context.scene
        frame_start = scene.frame_start
        frame_end = scene.frame_end

        # Extract camera intrinsic parameters
        intrinsic_params = self.get_camera_intrinsics(cam, scene)
        (focal_length_px, principal_x, principal_y,
         focal_length_mm, sensor_width_mm, sensor_height_mm) = intrinsic_params

        # Compute 35mm equivalent focal length
        crop_factor = 36.0 / sensor_width_mm
        focal_length_35mm = focal_length_mm * crop_factor

        # Distortion coefficients (assuming zero unless specified)
        K1 = K2 = K3 = P1 = P2 = P3 = 0.0

        # Get the render filepath components
        base_name, ext, base_dir = self.get_render_filepath_components(scene)

        # Use the specified output directory if provided
        output_dir = bpy.path.abspath(self.output_directory) if self.output_directory else base_dir

        # Ensure the output directory exists
        if not os.path.exists(output_dir):
            os.makedirs(output_dir)

        # Loop over the frames
        for frame in range(frame_start, frame_end + 1):
            scene.frame_set(frame)

            # Construct the image filename based on the render filepath
            image_name = self.get_image_name(base_name, ext, frame)

            #source_transform = cam.matrix_world.copy()
            source_transform = cam.matrix_local.copy()
            # Get camera extrinsic parameters
            cam_location = source_transform.to_translation()

            # Get camera rotation matrix
            cam_rotation_world = source_transform.to_3x3()

            # Convert to RealityCapture coordinate system
            rc_position = self.blender_to_rc_position(cam_location)
            rc_rotation_matrix = self.blender_to_rc_rotation(cam_rotation_world)

            # Flatten rotation matrix into a single line
            rotation_elements = [rc_rotation_matrix[row][col] for row in range(3) for col in range(3)]
            rotation_str = ' '.join(f"{elem:.15f}" for elem in rotation_elements)

            # Create the XMP file path
            xmp_filename = os.path.splitext(image_name)[0] + ".xmp"
            xmp_filepath = os.path.join(output_dir, xmp_filename)

            # Write XMP file
            with open(xmp_filepath, 'w') as f:
                f.write('<?xpacket begin="﻿" id="W5M0MpCehiHzreSzNTczkc9d"?>\n')
                f.write('<x:xmpmeta xmlns:x="adobe:ns:meta/">\n')
                f.write(' <rdf:RDF xmlns:rdf="http://www.w3.org/1999/02/22-rdf-syntax-ns#">\n')
                f.write('  <rdf:Description\n')
                f.write('   xcr:Version="3"\n')
                f.write('   xcr:PosePrior="initial"\n')
                f.write('   xcr:Coordinates="absolute"\n')
                f.write('   xcr:DistortionModel="brown3"\n')
                f.write(f'   xcr:FocalLength35mm="{focal_length_35mm:.15f}"\n')
                f.write('   xcr:Skew="0"\n')
                f.write('   xcr:AspectRatio="1"\n')
                f.write('   xcr:PrincipalPointU="0"\n')  # Assuming principal point offsets are zero
                f.write('   xcr:PrincipalPointV="0"\n')
                f.write('   xcr:CalibrationPrior="initial"\n')
                f.write('   xcr:CalibrationGroup="-1"\n')
                f.write('   xcr:DistortionGroup="-1"\n')
                f.write('   xcr:InTexturing="0"\n')
                f.write('   xcr:InMeshing="1"\n')
                f.write('   xmlns:xcr="http://www.capturingreality.com/ns/xcr/1.1#">\n')
                f.write(f'   <xcr:Rotation>{rotation_str}</xcr:Rotation>\n')
                f.write(f'   <xcr:Position>{rc_position.x:.15f} {rc_position.y:.15f} {rc_position.z:.15f}</xcr:Position>\n')
                f.write(f'   <xcr:DistortionCoeficients>{K1} {K2} {K3} {P1} {P2} {P3}</xcr:DistortionCoeficients>\n')
                f.write('  </rdf:Description>\n')
                f.write(' </rdf:RDF>\n')
                f.write('</x:xmpmeta>\n')
                f.write('<?xpacket end="w"?>\n')

        self.report({'INFO'}, f"Camera poses exported to XMP files in {output_dir}")
        return {'FINISHED'}
    
    def get_camera_intrinsics(self, cam, scene):
        # Camera intrinsic parameters
        cam_data = cam.data
        focal_length_mm = cam_data.lens  # Focal length in mm
        sensor_width_mm = cam_data.sensor_width  # Sensor width in mm
        sensor_height_mm = cam_data.sensor_height  # Sensor height in mm

        # Assuming images are rendered at the scene's resolution
        resolution_x_px = scene.render.resolution_x
        resolution_y_px = scene.render.resolution_y
        pixel_aspect = scene.render.pixel_aspect_x / scene.render.pixel_aspect_y

        # Adjust sensor size for pixel aspect ratio
        sensor_width_mm *= pixel_aspect

        # Principal point (assuming center of the image)
        principal_x = resolution_x_px / 2
        principal_y = resolution_y_px / 2

        # Focal length in pixels
        focal_length_px = (focal_length_mm / sensor_width_mm) * resolution_x_px

        return (focal_length_px, principal_x, principal_y,
                focal_length_mm, sensor_width_mm, sensor_height_mm)

    def get_render_filepath_components(self, scene):
        # Get the base filepath from the render settings
        render_filepath = bpy.path.abspath(scene.render.filepath)
        base_dir = os.path.dirname(render_filepath)
        base_name = os.path.basename(render_filepath)
        base_name, ext = os.path.splitext(base_name)

        # If no extension is provided, derive it from the render file format
        if not ext:
            ext = self.get_extension_from_format(scene.render.image_settings.file_format)

        # Handle frame placeholders in the base name
        base_name = self.process_base_name(base_name)

        return base_name, ext, base_dir

    def get_extension_from_format(self, file_format):
        # Map Blender's file formats to common extensions
        format_to_ext = {
            'BMP': '.bmp',
            'IRIS': '.rgb',
            'PNG': '.png',
            'JPEG': '.jpg',
            'JPEG2000': '.jp2',
            'TARGA': '.tga',
            'TARGA_RAW': '.tga',
            'CINEON': '.cin',
            'DPX': '.dpx',
            'OPEN_EXR_MULTILAYER': '.exr',
            'OPEN_EXR': '.exr',
            'HDR': '.hdr',
            'TIFF': '.tif',
            'AVI_JPEG': '.avi',
            'AVI_RAW': '.avi',
            'FRAMESERVER': '.fs',
            'FFMPEG': '.mp4',  # Defaulting to .mp4 for FFMPEG
        }
        return format_to_ext.get(file_format, '')

    def process_base_name(self, base_name):
        # If the base name contains a frame placeholder, we need to handle it
        if "%d" in base_name or "%0" in base_name:
            # Do nothing; the frame number will be inserted automatically
            pass
        elif "#" in base_name:
            # Do nothing; the frame number will be inserted automatically
            pass
        else:
            # If no frame number placeholder, append one
            base_name += "%04d"

        return base_name

    def get_image_name(self, base_name, ext, frame_number):
        image_name = base_name
        # Replace frame placeholders with actual frame numbers
        if "%" in image_name:
            image_name = image_name % frame_number
        elif "#" in image_name:
            num_hashes = image_name.count("#")
            image_name = image_name.replace("#" * num_hashes, f"{frame_number:0{num_hashes}d}")
        else:
            # Append frame number if no placeholder is present
            image_name += f"{frame_number:04d}"

        # Add the file extension
        image_name += ext

        return image_name

    #def blender_to_rc_position(self, position):
    #    # Convert Blender's coordinate system to RealityCapture's
    #    return mathutils.Vector((position.x, position.z, -position.y))

    def blender_to_rc_position(self, position):
        # If coordinate systems align, no change is needed
        return position.copy()

    #def blender_to_rc_rotation(self, rotation_matrix):
    #    # Swap Y and Z axes, invert Z
    #    swap_matrix = mathutils.Matrix(((1, 0, 0),
    #                                    (0, 0, 1),
    #                                    (0, -1, 0)))
    #    return swap_matrix @ rotation_matrix @ swap_matrix.inverted()

    def blender_to_rc_rotation(self, rotation_matrix):
        # Rotate 180 degrees around X-axis to flip Y and Z
        flip_matrix = mathutils.Matrix.Rotation(math.radians(180), 4, 'X').to_3x3()
        return flip_matrix @ rotation_matrix

    #def draw(self, context):
    #    layout = self.layout
    #    layout.prop(self, "output_directory")

    def invoke(self, context, event):
        wm = context.window_manager
        return wm.invoke_props_dialog(self)
        

        """ 

            Old bi state class
class AnimateCameraAlongFaceNormalsOperator(bpy.types.Operator):
     Animate a Single Camera Along Face Normals of Selected Meshes 
    bl_idname = "object.animate_camera_along_face_normals"
    bl_label = "Animate Camera Along Face Normals"
    bl_options = {'REGISTER', 'UNDO'}

    focal_length: bpy.props.FloatProperty(
        name="Focal Length",
        description="Focal length of the camera in mm",
        default=30.0
    )

    face_inward: bpy.props.BoolProperty(
        name="Face Inward",
        description="If True, the camera will face inward along the face normal; if False, it will face outward.",
        default=True
    )

    start_frame: bpy.props.IntProperty(
        name="Start Frame",
        description="Frame to start the animation",
        default=1
    )

    frame_step: bpy.props.IntProperty(
        name="Frames per Position",
        description="Number of frames to hold each camera position",
        default=1,
        min=1
    )

    def execute(self, context):
        selected_objects = context.selected_objects

        if not selected_objects:
            self.report({'ERROR'}, "No objects selected.")
            return {'CANCELLED'}

        mesh_objects = [obj for obj in selected_objects if obj.type == 'MESH']

        if not mesh_objects:
            self.report({'ERROR'}, "No mesh objects selected.")
            return {'CANCELLED'}

        scene = context.scene

        # Collect face centers and normals
        face_data = []
        for obj in mesh_objects:
            # Ensure the mesh data is up-to-date
            depsgraph = context.evaluated_depsgraph_get()
            obj_eval = obj.evaluated_get(depsgraph)
            mesh = obj_eval.to_mesh()

            # Transform the mesh to local coordinates
            mesh.transform(obj.matrix_local)

            # For each face in the mesh
            for poly in mesh.polygons:
                face_center = poly.center.copy()
                face_normal = poly.normal.copy()
                face_data.append((face_center, face_normal))

            # Clean up the evaluated mesh
            obj_eval.to_mesh_clear()

            # We don't want the object we are only using to set camera positions to show in renders
            # Turn off render visibility for this mesh object
            obj.hide_render = True

        if not face_data:
            self.report({'ERROR'}, "No faces found in the selected meshes.")
            return {'CANCELLED'}

        # Sort or shuffle face_data if desired
        # For example, to randomize the order:
        # import random
        # random.shuffle(face_data)

        # Create or get the camera
        cam = None
        # Create a new camera
        cam_data = bpy.data.cameras.new("SplatCamera")
        cam_data.lens = self.focal_length
        cam = bpy.data.objects.new("SplatCamera", cam_data)
        scene.collection.objects.link(cam)
        scene.camera = cam  # Set the new camera as the active camera
        cam.parent = mesh_objects[0].parent

        frame_number = self.start_frame

        # Animate the camera
        for idx, (face_center, face_normal) in enumerate(face_data):
            # Set the frame
            scene.frame_set(frame_number)

            # Set camera location to face center
            cam.location = face_center
            cam.keyframe_insert(data_path="location", index=-1)

            # Adjust face normal based on the 'face_inward' property
            if self.face_inward:
                direction = face_normal
            else:
                direction = -face_normal  # Reverse the normal for outward facing

            # Orient the camera to point along the adjusted face normal
            # Since the camera looks along its negative Z-axis, we need to adjust the rotation
            rot_quat = direction.to_track_quat('-Z', 'Y')
            cam.rotation_mode = 'QUATERNION'
            cam.rotation_quaternion = rot_quat
            cam.keyframe_insert(data_path="rotation_quaternion", index=-1)

            # Optionally, insert keyframes for focal length if needed
            cam.data.lens = self.focal_length
            cam.data.keyframe_insert(data_path="lens", frame=frame_number)

            # Move to the next frame
            frame_number += self.frame_step

        # Update the scene's frame range
        scene.frame_end = frame_number - self.frame_step

        self.report({'INFO'}, f"Camera animated along {len(face_data)} face normals.")
        return {'FINISHED'}

    def invoke(self, context, event):
        return context.window_manager.invoke_props_dialog(self)

    def draw(self, context):
        layout = self.layout
        layout.prop(self, "focal_length")
        layout.prop(self, "face_inward")
        #layout.prop(self, "start_frame")
        #layout.prop(self, "frame_step")

    def invoke(self, context, event):
        return context.window_manager.invoke_props_dialog(self)
    """
        
class AnimateCameraAlongFaceNormalsOperator(bpy.types.Operator):
    """Animate a Single Camera Along Face Normals of Selected Meshes"""
    bl_idname = "object.animate_camera_along_face_normals"
    bl_label = "Animate Camera Along Face Normals"
    bl_options = {'REGISTER', 'UNDO'}

    focal_length: bpy.props.FloatProperty(
        name="Focal Length",
        description="Focal length of the camera in mm",
        default=30.0
    )

    direction_mode: bpy.props.EnumProperty(
        name="Direction Mode",
        description="Choose camera direction along face normals",
        items=[
            ('INWARD', "Inward", "Camera faces inward along face normals"),
            ('OUTWARD', "Outward", "Camera faces outward along face normals"),
            ('BOTH', "Both", "Camera faces both inward and outward along face normals")
        ],
        default='BOTH'
    )
    
    def execute(self, context):
        selected_objects = context.selected_objects

        if not selected_objects:
            self.report({'ERROR'}, "No objects selected.")
            return {'CANCELLED'}

        mesh_objects = [obj for obj in selected_objects if obj.type == 'MESH']

        if not mesh_objects:
            self.report({'ERROR'}, "No mesh objects selected.")
            return {'CANCELLED'}

        scene = context.scene

        # Collect face centers and normals
        face_data = []
        for obj in mesh_objects:
            # Ensure the mesh data is up-to-date
            depsgraph = context.evaluated_depsgraph_get()
            obj_eval = obj.evaluated_get(depsgraph)
            mesh = obj_eval.to_mesh()

            # Transform the mesh to local coordinates
            mesh.transform(obj.matrix_local)

            # For each face in the mesh
            for poly in mesh.polygons:
                face_center = poly.center.copy()
                face_normal = poly.normal.copy()
                face_data.append((face_center, face_normal))

            # Clean up the evaluated mesh
            obj_eval.to_mesh_clear()

            # Turn off render visibility for this mesh object
            obj.hide_render = True

        if not face_data:
            self.report({'ERROR'}, "No faces found in the selected meshes.")
            return {'CANCELLED'}

        # Create a new camera
        cam_data = bpy.data.cameras.new("SplatCamera")
        cam_data.lens = self.focal_length
        cam = bpy.data.objects.new("SplatCamera", cam_data)
        scene.collection.objects.link(cam)
        scene.camera = cam  # Set the new camera as the active camera
        cam.parent = mesh_objects[0].parent if mesh_objects[0].parent else None

        frame_number = 1

        # Determine the directions based on the selected mode
        if self.direction_mode == 'INWARD':
            directions = ['INWARD']
        elif self.direction_mode == 'OUTWARD':
            directions = ['OUTWARD']
        elif self.direction_mode == 'BOTH':
            directions = ['OUTWARD', 'INWARD']  # Process OUTWARD first
        else:
            directions = []

        # Animate the camera
        for dir_mode in directions:
            for idx, (face_center, face_normal) in enumerate(face_data):
                # Set the frame
                scene.frame_set(frame_number)

                # Set camera location to face center
                cam.location = face_center
                cam.keyframe_insert(data_path="location", index=-1)

                # Adjust face normal based on the 'direction_mode'
                if dir_mode == 'INWARD':
                    direction = face_normal
                elif dir_mode == 'OUTWARD':
                    direction = -face_normal  # Reverse the normal for outward facing
                else:
                    continue  # Should not reach here

                # Orient the camera to point along the adjusted face normal
                # Since the camera looks along its negative Z-axis, adjust accordingly
                rot_quat = direction.to_track_quat('-Z', 'Y')
                cam.rotation_mode = 'QUATERNION'
                cam.rotation_quaternion = rot_quat
                cam.keyframe_insert(data_path="rotation_quaternion", index=-1)

                # Optionally, insert keyframes for focal length if needed
                cam.data.lens = self.focal_length
                cam.data.keyframe_insert(data_path="lens", frame=frame_number)

                # Move to the next frame
                frame_number += 1

        # Update the scene's frame range
        scene.frame_end = frame_number - 1

        self.report({'INFO'}, f"Camera animated along {len(face_data) * len(directions)} face normals.")
        return {'FINISHED'}

    def invoke(self, context, event):
        return context.window_manager.invoke_props_dialog(self)

    def draw(self, context):
        layout = self.layout
        layout.prop(self, "focal_length")
        layout.prop(self, "direction_mode")



class GenerateCameraPatternMeshOperator(bpy.types.Operator):
    """Generate Room Camera Rings Pattern Mesh"""
    bl_idname = "object.generate_camera_pattern_mesh_operator"
    bl_label = "Generate Camera Pattern Mesh"
    bl_options = {'REGISTER', 'UNDO'}

    radius: bpy.props.FloatProperty(
        name="Radius",
        description="Radius of the circles",
        default=1.0
    )

    def execute(self, context): 
        # Parameters
        N = 60  # Number of positions along each circle
        # radius is now a property: self.radius
        face_size = 0.1  # Size of each face (edge length)

        # Define the circle parameters (height and pitch angle)
        # Each entry is (height, pitch angle in degrees)
        circles = [
            (1.0, -20.0),   # Circle at 1 meter height, looking down 20 degrees
            (2.0, 0.0),     # Circle at 2 meters height, looking straight ahead
            (3.0, -20.0)    # Circle at 3 meters height, looking down 20 degrees
        ]
        
        scene = context.scene
        obj = context.active_object
        if obj is None:
            self.report({'ERROR'}, "No active object selected.")
            return {'CANCELLED'}
        
        face_half_size = face_size / 2.0  # Half size for convenience

        # Initialize lists for vertices and faces
        vertices = []
        faces = []
        vertex_index = 0  # To keep track of the vertex indices

        for height, pitch_adjust in circles:
            obj_location = mathutils.Vector((0, 0, height))

            # Calculate positions along the circle
            for i in range(N):
                angle = (2 * math.pi / N) * i
                x = self.radius * math.cos(angle)
                y = self.radius * math.sin(angle)
                z = height

                # Calculate the face location
                location = mathutils.Vector((x, y, z))

                # Compute rotation to look at the object center
                # Calculate the direction vector pointing from the object center to the face
                direction = (location - obj_location).normalized()

                # Create a rotation quaternion that aligns Z with the direction vector
                rotation_quat = direction.to_track_quat('Z', 'Y')

                # Create a pitch adjustment quaternion around the local X-axis
                pitch_quat = mathutils.Quaternion((1.0, 0.0, 0.0), math.radians(pitch_adjust))

                # Combine the pitch adjustment with the initial rotation
                rotation_quat = rotation_quat @ pitch_quat  # Apply pitch after aligning to direction

                # Get the rotation matrix from the adjusted quaternion
                rotation_matrix = rotation_quat.to_matrix()

                # Define local coordinates of the face vertices (centered at origin)
                local_verts = [
                    mathutils.Vector((-face_half_size, -face_half_size, 0)),
                    mathutils.Vector((face_half_size, -face_half_size, 0)),
                    mathutils.Vector((face_half_size, face_half_size, 0)),
                    mathutils.Vector((-face_half_size, face_half_size, 0)),
                ]

                # Transform vertices to world coordinates
                transformed_verts = []
                for v in local_verts:
                    # Apply rotation
                    world_v = rotation_matrix @ v
                    # Translate to location
                    world_v += location
                    transformed_verts.append(world_v)

                # Add vertices to the list
                vertices.extend(transformed_verts)

                # Define the face using the indices of the new vertices
                # Reverse the order to flip the normal inward
                face = [
                    vertex_index + 3,
                    vertex_index + 2,
                    vertex_index + 1,
                    vertex_index
                ]
                faces.append(face)

                vertex_index += 4  # Update the vertex index for the next face

        # Create a new mesh and object
        mesh = bpy.data.meshes.new("CameraPatternMesh")
        mesh.from_pydata(vertices, [], faces)
        mesh.update()

        mesh_obj = bpy.data.objects.new("CameraPatternMesh", mesh)
        scene.collection.objects.link(mesh_obj)

                # Parent the mesh object to the active object
        mesh_obj.parent = obj


        self.report({'INFO'}, "Camera pattern mesh generated successfully.")
        return {'FINISHED'}

    def invoke(self, context, event):
        return context.window_manager.invoke_props_dialog(self)

    def draw(self, context):
        layout = self.layout
        layout.prop(self, "radius")

class GenerateIcoSphereOperator(bpy.types.Operator):
    """Generate Generic ICO Sphere Camera Pattern Mesh"""
    bl_idname = "object.generate_icosphere_operator"
    bl_label = "Generate Icosphere Mesh"
    bl_options = {'REGISTER', 'UNDO'}

    radius: bpy.props.FloatProperty(
        name="Radius",
        description="Radius of the icosphere",
        default=2.0,
        min=0.01
    )

    subdivisions: bpy.props.IntProperty(
        name="Subdivisions",
        description="Number of subdivisions (detail level)",
        default=2,
        min=0,
        max=6
    )

    def execute(self, context):
        scene = context.scene

        # Create a new bmesh object
        bm = bmesh.new()

        # Create an icosphere using bmesh operations
        bmesh.ops.create_icosphere(
            bm,
            subdivisions=self.subdivisions,
            radius=self.radius
        )

        # Create a new mesh and link it to the scene
        mesh = bpy.data.meshes.new("IcosphereMesh")
        bm.to_mesh(mesh)
        bm.free()

        mesh_obj = bpy.data.objects.new("CameraIcosphere", mesh)
        scene.collection.objects.link(mesh_obj)

        # Set the object's location to Z=1
        mesh_obj.location.z = self.radius * 1.5

        # Optionally, parent the new object to the active object
        active_obj = context.active_object
        if active_obj:
            mesh_obj.parent = active_obj

        self.report({'INFO'}, "Icosphere generated successfully.")
        return {'FINISHED'}

    def invoke(self, context, event):
        return context.window_manager.invoke_props_dialog(self)

    def draw(self, context):
        layout = self.layout
        layout.prop(self, "radius")
        layout.prop(self, "subdivisions")

class RunRealityCapture(bpy.types.Operator):
    """Run RealityCapture with exported camera data"""
    bl_idname = "exportsplattrain.run_realitycapture"
    bl_label = "Run RealityCapture"
    bl_options = {'REGISTER', 'UNDO'}

    def execute(self, context):
        scene = context.scene

        # Extract the render filepath components to get base_dir
        base_dir = self.get_render_base_dir(scene)

        # Ensure the output directory exists
        if not os.path.exists(base_dir):
            os.makedirs(base_dir)

        # Path to this Python file's directory
        script_dir = os.path.dirname(os.path.abspath(__file__))
        bat_source = os.path.join(script_dir, "rc_v04.bat")
        if not os.path.exists(bat_source):
            self.report({'ERROR'}, f"Batch file not found: {bat_source}")
            return {'CANCELLED'}

        # Destination for the batch file
        bat_dest = os.path.join(base_dir, "rc_v04.bat")

        # Copy the batch file to the output directory
        shutil.copyfile(bat_source, bat_dest)

        # Change the current working directory to the batch file's directory and execute it
        command = f'cmd.exe /c start "RealityCapture" cmd.exe /k "{bat_dest}"'
        subprocess.Popen(command, shell=True, cwd=base_dir)

        self.report({'INFO'}, f"Running RealityCapture batch file in a new window: {bat_dest}")
        return {'FINISHED'}

    def get_render_base_dir(self, scene):
        # Get the base directory from the render filepath
        render_filepath = bpy.path.abspath(scene.render.filepath)
        base_dir = os.path.dirname(render_filepath)
        return base_dir
    
def register():
    bpy.utils.register_class(GenerateCameraPathOperator)
    bpy.utils.register_class(ExportCameraPosesOperator)
    bpy.utils.register_class(ExportCameraPosesToColmap)
    bpy.utils.register_class(ExportCameraPosesToXMP)
    bpy.utils.register_class(RunRealityCapture)
    bpy.utils.register_class(AnimateCameraAlongFaceNormalsOperator)
    bpy.utils.register_class(GenerateCameraPatternMeshOperator)
    bpy.utils.register_class(GenerateIcoSphereOperator)

def unregister():
    bpy.utils.unregister_class(GenerateCameraPathOperator)
    bpy.utils.unregister_class(ExportCameraPosesOperator) 
    bpy.utils.unregister_class(ExportCameraPosesToColmap) 
    bpy.utils.unregister_class(ExportCameraPosesToXMP)
    bpy.utils.unregister_class(RunRealityCapture)
    bpy.utils.unregister_class(AnimateCameraAlongFaceNormalsOperator)
    bpy.utils.unregister_class(GenerateCameraPatternMeshOperator)
    bpy.utils.unregister_class(GenerateIcoSphereOperator)
    

def adjust_to_colmap_format(source_transform):
    """
    Adjust Blender's camera transform to COLMAP's world-to-camera pose.
    """

    rot_blender = source_transform.to_quaternion()
    rot = mathutils.Quaternion((
        rot_blender.x,
        rot_blender.w,
        rot_blender.z,
        -rot_blender.y))
    #qw, qx, qy, qz = cam_rot.w, cam_rot.x, cam_rot.y, cam_rot.z

    T = source_transform.translation
    rotation_matrix = rot.to_matrix()
    T1 = -(rotation_matrix @ T)
    #tx, ty, tz = T1

    return T1, rot


    # Extract Blender's rotation and translation
    blender_translation = source_transform.translation
    blender_rotation = source_transform.to_quaternion()

    # Adjust rotation to COLMAP's convention
    colmap_rotation = mathutils.Quaternion((
        blender_rotation.w,
        blender_rotation.x,
        blender_rotation.z,
        -blender_rotation.y
    ))

    # Adjust translation axes for COLMAP's coordinate system
    adjusted_translation = mathutils.Vector((
        blender_translation.x,
        blender_translation.z,
        -blender_translation.y
    ))

    # Compute COLMAP translation (T) as -R * C
    rotation_matrix = colmap_rotation.to_matrix()
    colmap_translation = -(rotation_matrix @ adjusted_translation)

    return colmap_translation, colmap_rotation
