# ##### BEGIN GPL LICENSE BLOCK #####
#
#  This program is free software; you can redistribute it and/or
#  modify it under the terms of the GNU General Public License
#  as published by the Free Software Foundation; either version 2
#  of the License, or (at your option) any later version.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program; if not, write to the Free Software Foundation,
#  Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
#
# ##### END GPL LICENSE BLOCK #####
# Code copied from io_import_images_as_planes.py

# -----------------------------------------------------------------------------
# Image loading
import os
import warnings
import re
from itertools import count, repeat
from collections import namedtuple
from bpy_extras.image_utils import load_image
import bpy
import glob


ImageSpec = namedtuple(
    'ImageSpec',
    ['image', 'size', 'start_frame', 'frame_offset', 'timeline_start_frame','frame_duration'])

num_regex = re.compile('[0-9]')  # Find a single number
nums_regex = re.compile('[0-9]+')  # Find a set of numbers


def find_image_sequences(files):
    """From a group of files, detect image sequences.

    This returns a generator of tuples, which contain the filename,
    start frame, and length of the detected sequence

    >>> list(find_image_sequences([
    ...     "test2-001.jp2", "test2-002.jp2",
    ...     "test3-003.jp2", "test3-004.jp2", "test3-005.jp2", "test3-006.jp2",
    ...     "blaah"]))
    [('blaah', 1, 1), ('test2-001.jp2', 1, 2), ('test3-003.jp2', 3, 4)]

    """
    files = iter(sorted(files))
    prev_file = None
    pattern = ""
    matches = []
    segment = None
    length = 1
    for filename in files:
        new_pattern = num_regex.sub('#', filename)
        if new_pattern == filename: # for a file like desktop.ini
            continue 
        new_matches = list(map(int, nums_regex.findall(filename)))
        if new_pattern == pattern:
            # this file looks like it may be in sequence from the previous

            # if there are multiple sets of numbers, figure out what changed
            if segment is None:
                for i, prev, cur in zip(count(), matches, new_matches):
                    if prev != cur:
                        segment = i
                        break

            # did it only change by one?
            for i, prev, cur in zip(count(), matches, new_matches):
                if i == segment:
                    # We expect this to increment
                    prev = prev + length
                if prev != cur:
                    break

            # All good!
            else:
                length += 1
                continue

        # No continuation -> spit out what we found and reset counters
        if prev_file:
            if length > 1:
                yield prev_file, matches[segment], length
            else:
                yield prev_file, 1, 1

        prev_file = filename
        matches = new_matches
        pattern = new_pattern
        segment = None
        length = 1

    if prev_file:
        if length > 1:
            yield prev_file, matches[segment], length
        else:
            yield prev_file, 1, 1

def find_image_startswith(files,prefix,start_frame,timeline_start_frame):
    """
    From a group of files, find a file with a matching prefix that is a video
    """
    files = iter(sorted(files))
    for filename in files:
        if filename.startswith(prefix):
            l = filename.lower()
            base,ext = os.path.splitext(l)
            if ext == ".mp4" or ext == ".mov":
                yield filename, start_frame, timeline_start_frame


def load_imagesequence_fromdir(directory,start_frame,sequencestartframe,timeline_start_frame,prefix=None):
        images = tuple(load_images(
            os.listdir(directory),
            directory,
            force_reload=False,
            start_frame=start_frame,
            sequencestartframe=sequencestartframe,
            timeline_start_frame=timeline_start_frame,
            prefix=prefix
        ))
        return images

def load_imagestartswith(directory,filename,start_frame,timeline_start_frame):
        images = tuple(load_imagesstartswith(
            os.listdir(directory),
            directory,
            filename,
            force_reload=False,
            start_frame=start_frame,
            timeline_start_frame=timeline_start_frame
        ))
        return images


def load_images(filenames, directory, force_reload=False, start_frame=1,sequencestartframe=None,timeline_start_frame=1,prefix=None):
    """Wrapper for bpy's load_image

    Loads a set of images, movies, or even image sequences
    Returns a generator of ImageSpec wrapper objects later used for texture setup
    """
    
    # if finding sequences, we need some pre-processing first
    file_iter = find_image_sequences(filenames)


    for filename, offset, frames in file_iter:
        image = load_image(filename, directory, check_existing=True, force_reload=force_reload)
        if image is None:
            continue

        if prefix:
            #print(f"load_images Looking for {prefix} on {filename}")
            if not filename.startswith(prefix):
                print("skip")
                continue

        # Size is unavailable for sequences, so we grab it early
        size = tuple(image.size)

        if image.source == 'MOVIE':
            # Blender BPY BUG!
            # This number is only valid when read a second time in 2.77
            # This repeated line is not a mistake
            frames = image.frame_duration
            frames = image.frame_duration

        elif frames > 1:  # Not movie, but multiple frames -> image sequence
            image.source = 'SEQUENCE'

        yield ImageSpec(image, size, start_frame, sequencestartframe, timeline_start_frame, frames)

def load_imagesstartswith(filenames, directory, filename, force_reload=False, start_frame=1,timeline_start_frame=1):
    """Wrapper for bpy's load_image

    Loads an image if found that starts with filename. For textures like  UUID_cam.mp4
    Returns a generator of ImageSpec wrapper objects later used for texture setup
    """
    
    # if finding sequences, we need some pre-processing first
    file_iter = find_image_startswith(filenames,filename,start_frame,timeline_start_frame)


    for filename, offset, frames in file_iter:
        image = load_image(filename, directory, check_existing=True, force_reload=force_reload)

        # Size is unavailable for sequences, so we grab it early
        size = tuple(image.size)

        if image.source == 'MOVIE':
            # Blender BPY BUG!
            # This number is only valid when read a second time in 2.77
            # This repeated line is not a mistake
            frames = image.frame_duration
            frames = image.frame_duration

        elif frames > 1:  # Not movie, but multiple frames -> image sequence
            image.source = 'SEQUENCE'

        yield ImageSpec(image, size, start_frame,offset,timeline_start_frame, frames)
        break  # Only need 1

# Only called for video files
def load_imagesexact(filename, directory, force_reload=False, start_frame=1,timeline_start_frame=1):
    """Wrapper for bpy's load_image

    Returns a generator of ImageSpec wrapper objects later used for texture setup
    """
    

    image = load_image(filename, directory, check_existing=True, force_reload=force_reload)

    # Size is unavailable for sequences, so we grab it early
    size = tuple(image.size)

    if image.source == 'MOVIE':
        # Blender BPY BUG!
        # This number is only valid when read a second time in 2.77
        # This repeated line is not a mistake
        frames = image.frame_duration
        frames = image.frame_duration

    print(f"image.frame_duration {frames}")
    # I think the 1 should be total number of frames
    yield ImageSpec(image, size, start_frame,0,timeline_start_frame, image.frame_duration)


def pathprefix(s):
        i = s.rfind('_')
        if i >= 0:
            return s[0:i]
        return s

def cdupup(s):
    return os.path.dirname(os.path.dirname(s))


def TextureFind(texturedir:str,trackingpath:str,cameraname:str):
    """            
            # This will do a recursive search down from the texturedir which could be a relative blender path.
            # It takes matches in the following order
            #  1. cameraname matches directory name
            #  2. cameraname *_cam.mp4'
            #   This will match camera videos
            #   cameraname + '_cam.mp4'
            #
            #  Last if will look for dir/UUID_*.mp4 based on the tracking file path to find textures for single file imports

    Args:
        texturedir ([str]): [description]
        trackingpath ([str]): [description]  not used now that we are using parent shoot dir
        cameraname ([str]): [description]

    Returns:
        path,isdir
           path - None or valid path
           isdir - True if directory for image sequence 
    """
    if texturedir != None:
        tpath = bpy.path.abspath(texturedir)
        print("texturepath:",tpath)


        files = glob.glob(tpath + '/**/' + cameraname, recursive=True)

        if len(files) > 0:
            directory = files[0]
            if os.path.isdir(directory):
                return (directory,True)

        files = glob.glob(tpath + '/**/*' + cameraname + '*_cam.mp4', recursive=True)

        if len(files) > 0:
            path = files[0]
            if os.path.isfile(path):
                return (path, False)

    if trackingpath:
        pp = cdupup(trackingpath)
        # We might need to change this is _cam.mp4 or _comp.mp4 to eliminate depth matches
        files = glob.glob(pp + '/**/*' + cameraname + '*_cam.mp4', recursive=True)

        if len(files) > 0:
            path = files[0]
            if os.path.isfile(path):
                return (path, False)
           

    return (None,False)

def DepthVideoFind(texturedir:str,trackingpath:str,cameraname:str):
    """            
            # This will do a recursive search down from the texturedir which could be a relative blender path.
            # It takes matches in the following order
            #  2. cameraname *_depth.mp4'
            #  Last if will look for dir/UUID*_depth.mp4 based on the tracking file path to find textures for single file imports

    Args:
        texturedir ([str]): [description]
        trackingpath ([str]): [description]  not used now that we are using parent shoot dir
        cameraname ([str]): [description]

    Returns:
        path,isdir
           path - None or valid path
           isdir - True if directory for image sequence 
    """
    tpath = bpy.path.abspath(texturedir)
    print("texturepath:",tpath)


    files = glob.glob(tpath + '/**/*' + cameraname + '*_depth.mp4', recursive=True)

    if len(files) > 0:
        path = files[0]
        if os.path.isfile(path):
            return (path, False)

    pp = cdupup(trackingpath)
    # We might need to change this is _cam.mp4 or _comp.mp4 to eliminate depth matches
    files = glob.glob(pp + '/**/*' + cameraname + '*_depth.mp4', recursive=True)

    if len(files) > 0:
        path = files[0]
        if os.path.isfile(path):
            return (path, False)
           

    return (None,False)

def DepthAbcFind(lidardir:str,trackingpath:str,cameraname:str):
    """            
            #
            # This will do a recursive search down from the lidardir which could be a relative blender path.

            #  LIDAR search
            #  1. tpath + camera.name *.abc
            #
            #  Last if will look for dir/UUID_*.abc based on the tracking file path to find textures for single file imports


    Args:
        lidardir ([str]): [description]
        trackingpath ([str]): [not used now that we are using parent shoot dir]
        cameraname ([str]): [description]

    Returns:
        path
           path - None or valid path

    """
    tpath = bpy.path.abspath(lidardir)
    print("lidardir:",tpath)

    files = glob.glob(tpath + '/**/' + cameraname + '*.abc', recursive=True)

    if len(files) > 0:
        path = files[0]
        if os.path.isfile(path):
            return path

    files = glob.glob(cdupup(trackingpath) + '/**/' + cameraname + '*.abc', recursive=True)

    if len(files) > 0:
        path = files[0]
        if os.path.isfile(path):
            return path

    return None   