import bpy
import socket
import threading
import json
import mathutils
import math
import errno


class Lonet2CameraDriver:
    def __init__(self, ip='0.0.0.0', port=40000):
        self.ip = ip
        self.port = port
        self.sock = socket.socket(socket.AF_INET, socket.SOCK_DGRAM)
        self.sock.bind((self.ip, self.port))
        self.sock.settimeout(0.2)  # Set timeout to 100 milliseconds
        self.camera = None
        self.running = False
        self.thread = threading.Thread(target=self.listen)
        self.count = 0

        # Create the rotation quaternion for 90 degrees counterclockwise around the X-axis
        rotation_x = mathutils.Quaternion((math.cos(math.pi / 4), math.sin(math.pi / 4), 0, 0))

        # Create the rotation quaternion for 90 degrees counterclockwise around the Y-axis
        rotation_y = mathutils.Quaternion((math.cos(math.pi / 4), 0, math.sin(math.pi / 4), 0))

        # Create the rotation quaternion for 180 degrees around the Z-axis
        rotation_z = mathutils.Quaternion((math.cos(math.pi / 2), 0, 0, math.sin(math.pi / 2)))

        # Multiply the rotation quaternions with the existing quaternion
        # Note: Quaternion multiplication is not commutative, order matters.
        self.rotatetoblender = rotation_z @ rotation_x @ rotation_y

    def start(self):
        # Check if there's a camera named "jetset"
        camera_jetset = bpy.data.objects.get('jetset')
        if camera_jetset and camera_jetset.type == 'CAMERA':
            self.camera = camera_jetset
        else:
            # Find the first camera in the scene
            for obj in bpy.data.objects:
                if obj.type == 'CAMERA':
                    self.camera = obj
                    break

            # If no camera is found, create a new one
            if not self.camera:
                bpy.ops.object.camera_add()
                self.camera = bpy.context.object
                self.camera.name = 'jetset'

        if not self.camera:
            print("No camera found in the scene.")
            return False

        self.running = True
        print(f"Lonet2 reader started")

        self.thread.start()
        return True


    def stop(self):
        ret = self.running
        if self.running:
            self.running = False
            self.thread.join()
        if self.sock:
            self.sock.close()
            self.sock = None
        return ret

    def listen(self):
        while self.running:
            try:
                data, _ = self.sock.recvfrom(1024)
                if data:
                    self.process_data(data)
            except socket.timeout:
                continue

    def process_data(self, data):
        # Parse JSON data
        try:
            verbose = False
            self.count = self.count + 1
            if self.count % 100 == 0:
                self.count = 0
                verbose = False
            if verbose:
                print(f"process {data}")
            message = json.loads(data)
            if verbose:
                print(f"message {message}")
            if self.camera and 'camera_transform_data' in message:
                message = message['camera_transform_data']
                position = message['position']
                orientation = message['orientation']
                focal_length = message['focalLengthRaw']
                sensor_size = message['sensorSize']
                anim = -1
                if 'anim' in message:
                    anim = message['anim']

                # Transform position from Unreal to Blender
                blender_position = [position[0]*.01, -position[1]*.01, position[2]*.01]
                if verbose:
                    print(f"blender_position {blender_position}")

                x = orientation[1]
                y = orientation[2]
                z = -orientation[0]
                w = -orientation[3]
                blender_orientation = [w, x, y, z]
                if verbose:
                    print(f"blender_orientation {blender_orientation}")

            # Schedule the update on the main thread
            bpy.app.timers.register(lambda: self.update_camera(blender_position, blender_orientation, focal_length, sensor_size, anim))
        except Exception as e:
            print(f"Error processing data: {e}")

    def set_current_frame_from_time(self, time_in_seconds):
        # Get the current scene
        scene = bpy.context.scene
        
        # Get the start frame and frames per second (FPS) of the scene
        start_frame = scene.frame_start
        fps = scene.render.fps
        
        # Calculate the frame number from the time in seconds
        frame_number = start_frame + int(time_in_seconds * fps)
        
        # Check if the animation is currently playing
        if bpy.context.screen.is_animation_playing:
            # Stop the animation playback
            bpy.ops.screen.animation_play(False)

        # Set the current frame to the calculated frame number
        scene.frame_set(frame_number)

    def update_camera(self, position, orientation, focal_length, sensor_size, anim):
        if not self.camera:
            return  # In case the camera was not properly initialized

        # Update Blender camera
        

        if bpy.context.scene.AFTERBURNER.animationsync and anim >= 0:
            self.set_current_frame_from_time(anim)

        self.camera.location = mathutils.Vector(position)

        quaternion = mathutils.Quaternion(orientation)
        new_quaternion = self.rotatetoblender @ quaternion

        # Convert quaternion to Euler
        self.camera.rotation_euler = new_quaternion.to_euler()

        # Update focal length and sensor size
        self.camera.data.lens = focal_length
        self.camera.data.sensor_width = sensor_size[0]
        self.camera.data.sensor_height = sensor_size[1]

        return None  # Return None to unregister the timer

class StartLonet2Operator(bpy.types.Operator):
    bl_idname = "wm.start_lonet2"
    bl_label = "Start Lonet2"

    _driver = None

    def changebuttontext(self,text):
        from .utiloperators import LONET_PT_Panel

        LONET_PT_Panel.button_text = text
        for area in bpy.context.screen.areas:
            if area.type == 'VIEW_3D':
                area.tag_redraw()

    def execute(self, context):
        if StartLonet2Operator._driver is None:
            try:
                StartLonet2Operator._driver = Lonet2CameraDriver()
                ok = StartLonet2Operator._driver.start()
                if ok:
                    self.report({'INFO'}, "Lonet2 reader started ")
                    self.changebuttontext("Stop Lonet2")
            except OSError as e:
                if e.errno == errno.EADDRINUSE:
                    print("Error: Port is already in use.")
                    self.report({'INFO'}, "Error: Port is already in use.")
                else:
                    print(f"Unexpected OSError: {e}")
                    self.report({'INFO'}, "Error starting reader")
            except Exception as e:
                print(f"Unexpected error: {e}")
                self.report({'INFO'}, "Error starting reader")
        else:
            wasrunning = StartLonet2Operator._driver.stop()
            if wasrunning:
                self.report({'INFO'}, "Lonet2 reader stopped ")
            StartLonet2Operator._driver = None
            self.changebuttontext("Start Lonet2")
        return {'FINISHED'}


def menu_func(self, context):
    self.layout.operator(StartLonet2Operator.bl_idname)
    #self.layout.operator(StopLonet2Operator.bl_idname)

def register():
    bpy.utils.register_class(StartLonet2Operator)
    #bpy.utils.register_class(StopLonet2Operator)
    #bpy.types.VIEW3D_MT_view.append(menu_func)

def unregister():
    bpy.utils.unregister_class(StartLonet2Operator)
    #bpy.utils.unregister_class(StopLonet2Operator)
    #bpy.types.VIEW3D_MT_view.remove(menu_func)


